# FITUR AUTOCOMPLETE NOMOR PERKARA

## Deskripsi
Fitur autocomplete untuk input nomor perkara yang secara real-time mencari dan menampilkan suggestions dari database arsip SIPP. User dapat mengetik sebagian nomor perkara dan sistem akan menampilkan daftar nomor perkara yang cocok beserta informasi lokasi arsipnya.

## Fitur Utama

### 1. Real-time Search
- Pencarian dimulai setelah user mengetik minimal 2 karakter
- Debouncing 300ms untuk mengurangi request ke server
- Maksimal 10 hasil suggestion ditampilkan

### 2. Keyboard Navigation
- **Arrow Up/Down**: Navigasi antar suggestion
- **Enter**: Pilih suggestion yang di-highlight
- **Escape**: Tutup dropdown suggestion

### 3. Data Lengkap
Setiap suggestion menampilkan:
- **Nomor Perkara** (highlighted sesuai pencarian)
- **Nomor Arsip**
- **Lokasi Lengkap**: Ruang, Lemari, Rak, Berkas

### 4. Auto-fill Form
Ketika user memilih suggestion, form otomatis terisi:
- Nomor Perkara
- Nomor Arsip  
- No. Ruang
- No. Lemari
- No. Rak
- No. Berkas

## Struktur File

### 1. API Endpoint
**File**: `api/search-nomor-perkara.php`
- Endpoint untuk pencarian nomor perkara
- Koneksi ke database SIPP
- Return JSON array dengan hasil pencarian

### 2. JavaScript Component
**File**: `js/nomor-perkara-autocomplete.js`
- Class `NomorPerkaraAutocomplete` yang reusable
- Auto-inject CSS styles
- Event handling dan keyboard navigation

### 3. Demo Page
**File**: `demo-autocomplete.php`
- Contoh implementasi lengkap
- Form dengan multiple fields
- Integration example

## Cara Penggunaan

### 1. Basic Implementation
```html
<input type="text" id="nomor_perkara" placeholder="Nomor Perkara">

<script src="js/nomor-perkara-autocomplete.js"></script>
<script>
const autocomplete = new NomorPerkaraAutocomplete('#nomor_perkara');
</script>
```

### 2. Advanced Implementation
```javascript
const autocomplete = new NomorPerkaraAutocomplete('#nomor_perkara', {
    minLength: 3,           // Minimal karakter untuk pencarian
    debounceDelay: 500,     // Delay sebelum pencarian
    showDetails: true,      // Tampilkan detail lokasi
    onSelect: function(data) {
        // Callback ketika user memilih suggestion
        console.log('Selected:', data);
        
        // Auto-fill other fields
        document.getElementById('nomor_arsip').value = data.nomor_arsip;
        document.getElementById('no_ruang').value = data.no_ruang;
    }
});
```

### 3. Event Handling
```javascript
// Listen for selection event
document.getElementById('nomor_perkara').addEventListener('autocomplete:select', function(e) {
    const selectedData = e.detail;
    console.log('Nomor perkara selected:', selectedData);
});
```

## API Response Format

### Request
```
GET api/search-nomor-perkara.php?q=123
```

### Response
```json
[
    {
        "nomor_perkara": "123/Pid.B/2025/PN.Wtp",
        "nomor_arsip": "06",
        "no_ruang": "19",
        "no_lemari": "05", 
        "no_rak": "04",
        "no_berkas": "587"
    },
    {
        "nomor_perkara": "123/PDT.G/2025/PN.Wtp",
        "nomor_arsip": "07",
        "no_ruang": "20",
        "no_lemari": "06",
        "no_rak": "05", 
        "no_berkas": "588"
    }
]
```

## CSS Classes

### Container
```css
.autocomplete-container {
    position: relative;
}
```

### Suggestions Dropdown
```css
.autocomplete-suggestions {
    position: absolute;
    top: 100%;
    background: white;
    border: 1px solid #ddd;
    max-height: 300px;
    overflow-y: auto;
    z-index: 1000;
}
```

### Individual Suggestion
```css
.autocomplete-suggestion {
    padding: 10px 15px;
    cursor: pointer;
    border-bottom: 1px solid #f0f0f0;
}

.autocomplete-suggestion:hover,
.autocomplete-suggestion.selected {
    background-color: #f8f9fa;
}
```

## Keamanan & Performance

### 1. Security
- ✅ Input sanitization di API endpoint
- ✅ SQL injection prevention dengan prepared statements
- ✅ Authentication check pada API
- ✅ XSS prevention dengan proper escaping

### 2. Performance
- ✅ Debouncing untuk mengurangi request
- ✅ Limit 10 hasil untuk response cepat
- ✅ Lazy loading CSS styles
- ✅ Event delegation untuk memory efficiency

### 3. UX Improvements
- ✅ Loading indicator saat search
- ✅ No results message
- ✅ Error handling dengan graceful fallback
- ✅ Keyboard navigation
- ✅ Highlight matching text

## Browser Compatibility
- ✅ Chrome 60+
- ✅ Firefox 55+
- ✅ Safari 12+
- ✅ Edge 79+

## Dependencies
- Bootstrap 5.3+ (optional, untuk styling)
- FontAwesome 6+ (optional, untuk icons)
- Modern browser dengan ES6+ support

## Integration Examples

### 1. Form Peminjaman Berkas
```javascript
new NomorPerkaraAutocomplete('#nomor_perkara', {
    onSelect: function(data) {
        // Isi field nomor perkara dan validasi arsip ada
        if (data.nomor_arsip) {
            showSuccess('Nomor perkara valid dan tersedia di arsip');
        }
    }
});
```

### 2. Form Pencarian Arsip
```javascript
new NomorPerkaraAutocomplete('#search_nomor', {
    showDetails: false,
    onSelect: function(data) {
        // Redirect ke detail arsip
        window.location.href = `arsip-detail.php?nomor=${data.nomor_perkara}`;
    }
});
```

## Files Created
- `api/search-nomor-perkara.php`: API endpoint
- `js/nomor-perkara-autocomplete.js`: JavaScript component
- `demo-autocomplete.php`: Demo implementation
- `test-autocomplete-nomor-perkara.php`: Test page

## Date Created
September 1, 2025

## Next Steps
1. Integrate ke form peminjaman berkas
2. Add caching untuk performance
3. Add fuzzy search untuk typo tolerance
4. Add recent searches history
