<?php
require 'db.php';
require 'auth.php';
require_login();

// Get notifications for overdue and due today
function getNotifications() {
    global $db;
    
    $today = date('Y-m-d');
    
    // Get overdue books (past due date)
    $overdueStmt = $db->prepare("
        SELECT id, nama, nomor_perkara, tanggal_kembali, 
               JULIANDAY(?) - JULIANDAY(tanggal_kembali) as days_overdue
        FROM peminjaman 
        WHERE status = 'dipinjam' AND tanggal_kembali < ?
        ORDER BY tanggal_kembali ASC
    ");
    $overdueStmt->execute([$today, $today]);
    $overdue = $overdueStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get books due today
    $dueTodayStmt = $db->prepare("
        SELECT id, nama, nomor_perkara, tanggal_kembali
        FROM peminjaman 
        WHERE status = 'dipinjam' AND tanggal_kembali = ?
        ORDER BY nama ASC
    ");
    $dueTodayStmt->execute([$today]);
    $dueToday = $dueTodayStmt->fetchAll(PDO::FETCH_ASSOC);
    
    return [
        'overdue' => $overdue,
        'due_today' => $dueToday,
        'total_alerts' => count($overdue) + count($dueToday)
    ];
}

// AJAX endpoint
if (isset($_GET['ajax'])) {
    header('Content-Type: application/json');
    echo json_encode(getNotifications());
    exit;
}

$notifications = getNotifications();
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Notifikasi - Sistem Peminjaman Berkas</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1000px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .header { border-bottom: 2px solid #e74c3c; padding-bottom: 15px; margin-bottom: 20px; }
        .nav { margin-bottom: 20px; }
        .nav a { display: inline-block; padding: 8px 15px; margin-right: 10px; background: #3498db; color: white; text-decoration: none; border-radius: 4px; }
        .nav a:hover { background: #2980b9; }
        .alert-section { margin-bottom: 30px; }
        .alert-header { background: #e74c3c; color: white; padding: 10px 15px; margin: 0; font-size: 16px; border-radius: 4px 4px 0 0; }
        .alert-header.warning { background: #f39c12; }
        .alert-content { border: 1px solid #e74c3c; border-top: none; border-radius: 0 0 4px 4px; }
        .alert-content.warning { border-color: #f39c12; }
        .alert-item { padding: 12px 15px; border-bottom: 1px solid #ecf0f1; display: flex; justify-content: space-between; align-items: center; }
        .alert-item:last-child { border-bottom: none; }
        .alert-info { flex-grow: 1; }
        .alert-actions { display: flex; gap: 8px; }
        .btn { padding: 6px 12px; border: none; border-radius: 3px; cursor: pointer; text-decoration: none; font-size: 12px; }
        .btn-primary { background: #3498db; color: white; }
        .btn-success { background: #27ae60; color: white; }
        .btn:hover { opacity: 0.8; }
        .no-alerts { padding: 20px; text-align: center; color: #7f8c8d; font-style: italic; }
        .badge { background: #e74c3c; color: white; padding: 2px 6px; border-radius: 10px; font-size: 11px; margin-left: 5px; }
        .days-overdue { color: #e74c3c; font-weight: bold; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div style="display: flex; align-items: center; margin-bottom: 10px;">
                <img src="gambar/Logo_pnwtp.png" alt="Logo PN Watampone" width="80" height="80" style="margin-right: 15px;">
                <div>
                    <h1>🔔 Notifikasi Peminjaman Berkas</h1>
                    <p>Pengadilan Negeri Watampone - Pantau berkas yang harus dikembalikan hari ini dan yang sudah terlambat</p>
                </div>
            </div>
        </div>
        
        <div class="nav">
            <a href="index.php">← Kembali ke Dashboard</a>
            <a href="javascript:location.reload()">🔄 Refresh</a>
        </div>

        <?php if ($notifications['total_alerts'] == 0): ?>
            <div class="no-alerts">
                ✅ Tidak ada berkas yang perlu perhatian khusus hari ini!
            </div>
        <?php else: ?>
            
            <?php if (!empty($notifications['overdue'])): ?>
            <div class="alert-section">
                <h3 class="alert-header">
                    🚨 BERKAS TERLAMBAT (<?= count($notifications['overdue']) ?> item)
                </h3>
                <div class="alert-content">
                    <?php foreach ($notifications['overdue'] as $item): ?>
                    <div class="alert-item">
                        <div class="alert-info">
                            <strong><?= htmlspecialchars($item['nama']) ?></strong><br>
                            <small>Perkara: <?= htmlspecialchars($item['nomor_perkara']) ?></small><br>
                            <small>Seharusnya kembali: <?= htmlspecialchars($item['tanggal_kembali']) ?></small>
                            <span class="days-overdue">(Terlambat <?= (int)$item['days_overdue'] ?> hari)</span>
                        </div>
                        <div class="alert-actions">
                            <a href="index.php?highlight=<?= $item['id'] ?>" class="btn btn-primary">Lihat Detail</a>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>

            <?php if (!empty($notifications['due_today'])): ?>
            <div class="alert-section">
                <h3 class="alert-header warning">
                    ⏰ JATUH TEMPO HARI INI (<?= count($notifications['due_today']) ?> item)
                </h3>
                <div class="alert-content warning">
                    <?php foreach ($notifications['due_today'] as $item): ?>
                    <div class="alert-item">
                        <div class="alert-info">
                            <strong><?= htmlspecialchars($item['nama']) ?></strong><br>
                            <small>Perkara: <?= htmlspecialchars($item['nomor_perkara']) ?></small><br>
                            <small>Harus dikembalikan: <?= htmlspecialchars($item['tanggal_kembali']) ?> (HARI INI)</small>
                        </div>
                        <div class="alert-actions">
                            <a href="index.php?highlight=<?= $item['id'] ?>" class="btn btn-primary">Lihat Detail</a>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>
            
        <?php endif; ?>
    </div>

    <script>
        // Auto refresh every 5 minutes
        setTimeout(function() {
            location.reload();
        }, 300000);
    </script>

    <!-- Footer -->
    <footer class="bg-primary text-white text-center py-3 mt-5">
      <div class="container">
        <p class="mb-0">©2025 by Pn Watampone, Dibuat oleh Fahrurrasyid, A.Md.T CPNS Mahkamah Agung RI Angkatan 2024</p>
      </div>
    </footer>

</body>
</html>
