<?php
// Handle registration form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $nama_lengkap = trim($_POST['nama_lengkap'] ?? '');
    $nik = trim($_POST['nik'] ?? '');
    $no_telepon = trim($_POST['no_telepon'] ?? '');
    $alamat = trim($_POST['alamat'] ?? '');
    
    $errors = [];
    
    // Validation
    if (empty($username)) $errors[] = 'Username harus diisi';
    if (strlen($username) < 3) $errors[] = 'Username minimal 3 karakter';
    if (empty($email)) $errors[] = 'Email harus diisi';
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) $errors[] = 'Email tidak valid';
    if (empty($password)) $errors[] = 'Password harus diisi';
    if (strlen($password) < 6) $errors[] = 'Password minimal 6 karakter';
    if ($password !== $confirm_password) $errors[] = 'Konfirmasi password tidak sama';
    if (empty($nama_lengkap)) $errors[] = 'Nama lengkap harus diisi';
    if (empty($nik)) $errors[] = 'NIK harus diisi';
    if (strlen($nik) !== 16) $errors[] = 'NIK harus 16 digit';
    if (!ctype_digit($nik)) $errors[] = 'NIK harus berupa angka';
    if (empty($no_telepon)) $errors[] = 'Nomor telepon harus diisi';
    if (empty($alamat)) $errors[] = 'Alamat harus diisi';
    
    // Check if username already exists
    if (empty($errors)) {
        $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
        $stmt->execute([$username, $email]);
        if ($stmt->fetch()) {
            $errors[] = 'Username atau email sudah terdaftar';
        }
        
        // Check if NIK already exists
        $stmt = $pdo->prepare("SELECT id FROM pemohon_profiles WHERE nik = ?");
        $stmt->execute([$nik]);
        if ($stmt->fetch()) {
            $errors[] = 'NIK sudah terdaftar';
        }
    }
    
    // Handle file upload
    $ktpPath = null;
    if (empty($errors) && isset($_FILES['ktp_file']) && $_FILES['ktp_file']['error'] === 0) {
        $uploadFile = $_FILES['ktp_file'];
        $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'application/pdf'];
        $maxSize = 5 * 1024 * 1024; // 5MB
        
        if (!in_array($uploadFile['type'], $allowedTypes)) {
            $errors[] = 'File KTP harus berformat JPG, PNG, atau PDF';
        } elseif ($uploadFile['size'] > $maxSize) {
            $errors[] = 'Ukuran file KTP maksimal 5MB';
        } else {
            // Create upload directory if not exists
            $uploadDir = '../writable/uploads/ktp/';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0755, true);
            }
            
            $fileExtension = pathinfo($uploadFile['name'], PATHINFO_EXTENSION);
            $fileName = 'ktp_' . uniqid() . '_' . time() . '.' . $fileExtension;
            $ktpPath = $uploadDir . $fileName;
            
            if (!move_uploaded_file($uploadFile['tmp_name'], $ktpPath)) {
                $errors[] = 'Gagal mengupload file KTP';
                $ktpPath = null;
            } else {
                $ktpPath = 'ktp/' . $fileName; // Store relative path
            }
        }
    } elseif (empty($errors)) {
        $errors[] = 'File KTP harus diupload';
    }
    
    // Insert to database if no errors
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();
            
            // Insert user
            $stmt = $pdo->prepare("INSERT INTO users (username, email, password, role, is_active, created_at, updated_at) VALUES (?, ?, ?, 'pemohon', 0, NOW(), NOW())");
            $stmt->execute([$username, $email, password_hash($password, PASSWORD_DEFAULT)]);
            $userId = $pdo->lastInsertId();
            
            // Insert pemohon profile
            $stmt = $pdo->prepare("INSERT INTO pemohon_profiles (user_id, nama_lengkap, nik, no_telepon, alamat, ktp_path, status_verified, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, 'pending', NOW(), NOW())");
            $stmt->execute([$userId, $nama_lengkap, $nik, $no_telepon, $alamat, $ktpPath]);
            
            $pdo->commit();
            
            setFlashMessage('success', 'Registrasi berhasil! Silakan tunggu verifikasi dari petugas.');
            redirect('?page=login');
            
        } catch (Exception $e) {
            $pdo->rollBack();
            if ($ktpPath && file_exists('../writable/uploads/' . $ktpPath)) {
                unlink('../writable/uploads/' . $ktpPath);
            }
            $errors[] = 'Gagal menyimpan data: ' . $e->getMessage();
        }
    }
    
    if (!empty($errors)) {
        setFlashMessage('error', implode('<br>', $errors));
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Registrasi Pemohon - Permohonan Salinan Putusan</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px 0;
        }
        .card-custom {
            border: none;
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
            backdrop-filter: blur(10px);
            background: rgba(255,255,255,0.95);
        }
        .step-header {
            border-bottom: 2px solid #e9ecef;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-8 col-lg-7">
                <div class="card card-custom">
                    <div class="card-body p-5">
                        <!-- Header -->
                        <div class="text-center mb-4">
                            <div class="mb-3">
                                <i class="fas fa-user-plus fa-3x text-primary"></i>
                            </div>
                            <h3 class="card-title fw-bold text-primary">Registrasi Pemohon</h3>
                            <p class="text-muted">Daftar untuk mengajukan permohonan salinan putusan</p>
                        </div>

                        <!-- Flash Messages -->
                        <?php if ($error = getFlashMessage('error')): ?>
                            <div class="alert alert-danger">
                                <i class="fas fa-exclamation-circle me-2"></i>
                                <?= $error ?>
                            </div>
                        <?php endif; ?>

                        <!-- Registration Form -->
                        <form method="POST" enctype="multipart/form-data" class="needs-validation" novalidate>
                            
                            <!-- Step 1: Account Information -->
                            <div class="mb-4">
                                <h5 class="text-primary mb-3 step-header">
                                    <i class="fas fa-user-circle me-2"></i>Informasi Akun
                                </h5>
                                
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="username" class="form-label">Username *</label>
                                        <input type="text" 
                                               class="form-control" 
                                               id="username" 
                                               name="username" 
                                               value="<?= htmlspecialchars($_POST['username'] ?? '') ?>" 
                                               placeholder="Username untuk login"
                                               required>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="email" class="form-label">Email *</label>
                                        <input type="email" 
                                               class="form-control" 
                                               id="email" 
                                               name="email" 
                                               value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" 
                                               placeholder="email@example.com"
                                               required>
                                    </div>
                                </div>

                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="password" class="form-label">Password *</label>
                                        <div class="input-group">
                                            <input type="password" 
                                                   class="form-control" 
                                                   id="password" 
                                                   name="password" 
                                                   placeholder="Minimal 6 karakter"
                                                   minlength="6"
                                                   required>
                                            <button class="btn btn-outline-secondary" type="button" id="togglePassword">
                                                <i class="fas fa-eye" id="eyeIcon"></i>
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="confirm_password" class="form-label">Konfirmasi Password *</label>
                                        <input type="password" 
                                               class="form-control" 
                                               id="confirm_password" 
                                               name="confirm_password" 
                                               placeholder="Ulangi password"
                                               required>
                                    </div>
                                </div>
                            </div>

                            <!-- Step 2: Personal Information -->
                            <div class="mb-4">
                                <h5 class="text-primary mb-3 step-header">
                                    <i class="fas fa-id-card me-2"></i>Informasi Pribadi
                                </h5>
                                
                                <div class="mb-3">
                                    <label for="nama_lengkap" class="form-label">Nama Lengkap *</label>
                                    <input type="text" 
                                           class="form-control" 
                                           id="nama_lengkap" 
                                           name="nama_lengkap" 
                                           value="<?= htmlspecialchars($_POST['nama_lengkap'] ?? '') ?>" 
                                           placeholder="Nama lengkap sesuai KTP"
                                           required>
                                </div>

                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="nik" class="form-label">NIK (Nomor Induk Kependudukan) *</label>
                                        <input type="text" 
                                               class="form-control" 
                                               id="nik" 
                                               name="nik" 
                                               value="<?= htmlspecialchars($_POST['nik'] ?? '') ?>" 
                                               placeholder="16 digit NIK"
                                               maxlength="16"
                                               pattern="[0-9]{16}"
                                               required>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="no_telepon" class="form-label">Nomor Telepon *</label>
                                        <input type="tel" 
                                               class="form-control" 
                                               id="no_telepon" 
                                               name="no_telepon" 
                                               value="<?= htmlspecialchars($_POST['no_telepon'] ?? '') ?>" 
                                               placeholder="081234567890"
                                               required>
                                    </div>
                                </div>

                                <div class="mb-3">
                                    <label for="alamat" class="form-label">Alamat Lengkap *</label>
                                    <textarea class="form-control" 
                                              id="alamat" 
                                              name="alamat" 
                                              rows="3" 
                                              placeholder="Alamat lengkap sesuai KTP"
                                              required><?= htmlspecialchars($_POST['alamat'] ?? '') ?></textarea>
                                </div>
                            </div>

                            <!-- Step 3: Document Upload -->
                            <div class="mb-4">
                                <h5 class="text-primary mb-3 step-header">
                                    <i class="fas fa-upload me-2"></i>Upload Dokumen
                                </h5>
                                
                                <div class="mb-3">
                                    <label for="ktp_file" class="form-label">File KTP *</label>
                                    <input type="file" 
                                           class="form-control" 
                                           id="ktp_file" 
                                           name="ktp_file" 
                                           accept=".pdf,.jpg,.jpeg,.png"
                                           required>
                                    <div class="form-text">
                                        Format yang diperbolehkan: PDF, JPG, JPEG, PNG. Maksimal 5MB.
                                    </div>
                                </div>

                                <!-- File Preview -->
                                <div id="file-preview" class="mt-3" style="display: none;">
                                    <div class="card border-success">
                                        <div class="card-body">
                                            <h6 class="card-title text-success">
                                                <i class="fas fa-check-circle me-2"></i>File Terpilih
                                            </h6>
                                            <p class="card-text mb-0" id="file-info"></p>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Terms and Conditions -->
                            <div class="mb-4">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="terms" name="terms" required>
                                    <label class="form-check-label" for="terms">
                                        Saya menyatakan bahwa data yang saya isi adalah <strong>benar dan dapat dipertanggungjawabkan</strong>. Saya memahami bahwa memberikan data palsu dapat dikenakan sanksi hukum.
                                    </label>
                                </div>
                            </div>

                            <!-- Submit Button -->
                            <div class="d-grid gap-2">
                                <button type="submit" class="btn btn-primary btn-lg">
                                    <i class="fas fa-user-plus me-2"></i> Daftar Sekarang
                                </button>
                            </div>

                        </form>

                        <!-- Login Link -->
                        <hr class="my-4">
                        <div class="text-center">
                            <p class="mb-0">Sudah memiliki akun? 
                                <a href="?page=login" class="text-decoration-none">
                                    <i class="fas fa-sign-in-alt me-1"></i>Login di sini
                                </a>
                            </p>
                        </div>

                        <!-- Back to Home -->
                        <div class="text-center mt-3">
                            <a href="setup.php" class="text-decoration-none">
                                <i class="fas fa-arrow-left me-1"></i> Kembali ke Beranda
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toggle password visibility
        document.getElementById('togglePassword').addEventListener('click', function() {
            const passwordField = document.getElementById('password');
            const eyeIcon = document.getElementById('eyeIcon');
            
            if (passwordField.type === 'password') {
                passwordField.type = 'text';
                eyeIcon.classList.remove('fa-eye');
                eyeIcon.classList.add('fa-eye-slash');
            } else {
                passwordField.type = 'password';
                eyeIcon.classList.remove('fa-eye-slash');
                eyeIcon.classList.add('fa-eye');
            }
        });

        // Password confirmation validation
        document.getElementById('confirm_password').addEventListener('input', function() {
            const password = document.getElementById('password').value;
            const confirmPassword = this.value;
            
            if (password !== confirmPassword) {
                this.setCustomValidity('Password tidak sama');
            } else {
                this.setCustomValidity('');
            }
        });

        // NIK validation (only numbers)
        document.getElementById('nik').addEventListener('input', function() {
            this.value = this.value.replace(/[^0-9]/g, '');
            if (this.value.length !== 16) {
                this.setCustomValidity('NIK harus terdiri dari 16 digit');
            } else {
                this.setCustomValidity('');
            }
        });

        // Phone number validation
        document.getElementById('no_telepon').addEventListener('input', function() {
            this.value = this.value.replace(/[^0-9]/g, '');
        });

        // File upload preview
        document.getElementById('ktp_file').addEventListener('change', function() {
            const file = this.files[0];
            const preview = document.getElementById('file-preview');
            const fileInfo = document.getElementById('file-info');
            
            if (file) {
                const fileSize = formatFileSize(file.size);
                const fileName = file.name;
                
                fileInfo.innerHTML = `
                    <strong>Nama File:</strong> ${fileName}<br>
                    <strong>Ukuran:</strong> ${fileSize}<br>
                    <strong>Tipe:</strong> ${file.type}
                `;
                preview.style.display = 'block';
                
                // Validate file size (5MB = 5 * 1024 * 1024 bytes)
                if (file.size > 5 * 1024 * 1024) {
                    this.setCustomValidity('Ukuran file tidak boleh lebih dari 5MB');
                } else {
                    this.setCustomValidity('');
                }
            } else {
                preview.style.display = 'none';
            }
        });

        // Format file size function
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }

        // Form validation
        (function() {
            'use strict';
            window.addEventListener('load', function() {
                var forms = document.getElementsByClassName('needs-validation');
                var validation = Array.prototype.filter.call(forms, function(form) {
                    form.addEventListener('submit', function(event) {
                        if (form.checkValidity() === false) {
                            event.preventDefault();
                            event.stopPropagation();
                        }
                        form.classList.add('was-validated');
                    }, false);
                });
            }, false);
        })();
    </script>
</body>
</html>
