<?php
// Check if session is not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ?page=login');
    exit;
}

// Use the main config file for database connection
require_once '../config.php';

// Create database connection using config
$pdo = getDbConnection();

// Helper functions only if not already defined
if (!function_exists('getBaseUrl')) {
    function getBaseUrl() {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'];
        $script = $_SERVER['SCRIPT_NAME'];
        $path = dirname(dirname($script)); // Go up one level from public/
        return $protocol . '://' . $host . $path;
    }
}

if (!function_exists('getImageUrl')) {
    function getImageUrl($imageName) {
        return getBaseUrl() . '/gambar/' . $imageName;
    }
}

// Get permohonan ID from URL
$permohonan_id = $_GET['id'] ?? null;

if (!$permohonan_id) {
    header('Location: ?page=permohonan-saya');
    exit;
}

// Get permohonan data with user info and pemohon profile
$stmt = $pdo->prepare("
    SELECT p.*, u.username, u.email, u.nama_lengkap as user_nama,
           pp.nama_lengkap, pp.alamat, pp.no_telepon, pp.nik
    FROM permohonans p 
    JOIN users u ON p.pemohon_id = u.id 
    LEFT JOIN pemohon_profiles pp ON pp.user_id = u.id
    WHERE p.id = ? AND p.pemohon_id = ?
");
$stmt->execute([$permohonan_id, $_SESSION['user_id']]);
$permohonan = $stmt->fetch();

if (!$permohonan) {
    header('Location: ?page=permohonan-saya');
    exit;
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Print Permohonan Salinan Putusan - <?php echo htmlspecialchars($permohonan['nomor_perkara']); ?></title>
    <style>
        @page {
            size: A4;
            margin: 2cm 2.5cm 2cm 2.5cm;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: "Times New Roman", serif;
            font-size: 12pt;
            line-height: 1.5;
            color: #000;
            background: white;
        }
        
        .print-container {
            width: 100%;
            max-width: 210mm;
            margin: 0 auto;
            background: white;
            padding: 0;
        }
        
        .kop-surat {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 3px solid #000;
            padding-bottom: 15px;
        }
        
        .kop-surat img {
            max-width: 100%;
            height: auto;
        }
        
        .title {
            text-align: center;
            font-weight: bold;
            font-size: 14pt;
            margin: 30px 0 20px 0;
            text-transform: uppercase;
            text-decoration: underline;
        }
        
        .content {
            margin: 20px 0;
            text-align: justify;
        }
        
        .data-permohonan {
            margin: 20px 0;
        }
        
        .data-row {
            display: flex;
            margin-bottom: 8px;
            page-break-inside: avoid;
        }
        
        .data-label {
            width: 200px;
            font-weight: normal;
            flex-shrink: 0;
        }
        
        .data-colon {
            width: 20px;
            flex-shrink: 0;
        }
        
        .data-value {
            flex: 1;
            font-weight: normal;
        }
        
        .signature-section {
            margin-top: 50px;
            display: flex;
            justify-content: space-between;
        }
        
        .signature-left, .signature-right {
            width: 45%;
        }
        
        .signature-date {
            text-align: right;
            margin-bottom: 80px;
        }
        
        .signature-name {
            text-align: center;
            margin-top: 80px;
            border-top: 1px solid #000;
            padding-top: 5px;
        }
        
        .note {
            margin-top: 30px;
            font-size: 10pt;
            font-style: italic;
        }
        
        @media print {
            body {
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
            }
            
            .no-print {
                display: none !important;
            }
            
            .print-container {
                box-shadow: none;
                margin: 0;
                max-width: none;
            }
        }
        
        .print-button {
            position: fixed;
            top: 20px;
            right: 20px;
            background: #007bff;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 14px;
            z-index: 1000;
        }
        
        .print-button:hover {
            background: #0056b3;
        }
        
        .back-button {
            position: fixed;
            top: 20px;
            left: 20px;
            background: #6c757d;
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 14px;
            z-index: 1000;
            text-decoration: none;
            display: inline-block;
        }
        
        .back-button:hover {
            background: #545b62;
            color: white;
            text-decoration: none;
        }
    </style>
</head>
<body>
    <a href="?page=permohonan-saya" class="back-button no-print">← Kembali</a>
    <button onclick="window.print()" class="print-button no-print">🖨️ Print</button>

    <div class="print-container">
        <!-- Kop Surat -->
        <div class="kop-surat">
            <img src="<?php echo getImageUrl('kop_surat.png'); ?>" alt="Kop Surat PN Watampone">
        </div>

        <!-- Title -->
        <div class="title">
            PERMOHONAN SALINAN PUTUSAN
        </div>

        <!-- Content -->
        <div class="content">
            <p>Yang bertanda tangan di bawah ini:</p>
        </div>

        <!-- Data Pemohon -->
        <div class="data-permohonan">
            <div class="data-row">
                <div class="data-label">Nama</div>
                <div class="data-colon">:</div>
                <div class="data-value"><?php echo htmlspecialchars($permohonan['nama_lengkap'] ?: $permohonan['user_nama'] ?: $permohonan['username']); ?></div>
            </div>
            <div class="data-row">
                <div class="data-label">NIK</div>
                <div class="data-colon">:</div>
                <div class="data-value"><?php echo htmlspecialchars($permohonan['nik'] ?: '-'); ?></div>
            </div>
            <div class="data-row">
                <div class="data-label">Alamat</div>
                <div class="data-colon">:</div>
                <div class="data-value"><?php echo htmlspecialchars($permohonan['alamat'] ?: '-'); ?></div>
            </div>
            <div class="data-row">
                <div class="data-label">No. Telepon</div>
                <div class="data-colon">:</div>
                <div class="data-value"><?php echo htmlspecialchars($permohonan['no_telepon'] ?: '-'); ?></div>
            </div>
            <div class="data-row">
                <div class="data-label">Email</div>
                <div class="data-colon">:</div>
                <div class="data-value"><?php echo htmlspecialchars($permohonan['email']); ?></div>
            </div>
        </div>

        <div class="content">
            <p>Dengan ini mengajukan permohonan untuk mendapatkan salinan putusan dengan data sebagai berikut:</p>
        </div>

        <!-- Data Perkara -->
        <div class="data-permohonan">
            <div class="data-row">
                <div class="data-label">Nomor Perkara</div>
                <div class="data-colon">:</div>
                <div class="data-value"><?php echo htmlspecialchars($permohonan['nomor_perkara']); ?></div>
            </div>
            <div class="data-row">
                <div class="data-label">Para Pihak</div>
                <div class="data-colon">:</div>
                <div class="data-value"><?php echo htmlspecialchars($permohonan['para_pihak']); ?></div>
            </div>
            <div class="data-row">
                <div class="data-label">Jenis Perkara</div>
                <div class="data-colon">:</div>
                <div class="data-value"><?php echo htmlspecialchars($permohonan['jenis_perkara']); ?></div>
            </div>
            <div class="data-row">
                <div class="data-label">Kebutuhan</div>
                <div class="data-colon">:</div>
                <div class="data-value"><?php echo htmlspecialchars($permohonan['kebutuhan']); ?></div>
            </div>
            <div class="data-row">
                <div class="data-label">Tanggal Permohonan</div>
                <div class="data-colon">:</div>
                <div class="data-value"><?php echo date('d F Y', strtotime($permohonan['created_at'])); ?></div>
            </div>
            <div class="data-row">
                <div class="data-label">Status</div>
                <div class="data-colon">:</div>
                <div class="data-value"><?php echo htmlspecialchars($permohonan['status']); ?></div>
            </div>
        </div>

        <div class="content">
            <p>Salinan putusan tersebut diperlukan untuk kepentingan <?php echo strtolower(htmlspecialchars($permohonan['kebutuhan'])); ?>.</p>
            <p>Demikian permohonan ini saya sampaikan. Atas perhatian dan kerjasamanya, saya ucapkan terima kasih.</p>
        </div>

        <!-- Signature Section -->
        <div class="signature-section">
            <div class="signature-left">
                <!-- Kosong untuk tanda tangan petugas -->
            </div>
            <div class="signature-right">
                <div class="signature-date">
                    Watampone, <?php echo date('d F Y'); ?>
                </div>
                <div style="text-align: center; margin-bottom: 20px;">
                    <strong>Pemohon,</strong>
                </div>
                <div class="signature-name">
                    <strong><?php echo htmlspecialchars($permohonan['nama_lengkap'] ?: $permohonan['user_nama'] ?: $permohonan['username']); ?></strong>
                </div>
            </div>
        </div>

        <!-- Note -->
        <div class="note">
            <p><strong>Catatan:</strong></p>
            <p>- Permohonan ini dibuat melalui sistem online Pengadilan Negeri Watampone</p>
            <p>- Nomor referensi: <?php echo $permohonan['kode_permohonan']; ?></p>
            <p>- Dicetak pada: <?php echo date('d F Y H:i:s'); ?></p>
        </div>
    </div>

    <script>
        // Auto print when page loads (optional)
        // window.onload = function() {
        //     setTimeout(function() {
        //         window.print();
        //     }, 1000);
        // }
        
        // Print function
        function printDocument() {
            window.print();
        }
        
        // Handle print button
        document.addEventListener('DOMContentLoaded', function() {
            const printBtn = document.querySelector('.print-button');
            if (printBtn) {
                printBtn.addEventListener('click', printDocument);
            }
        });
    </script>
</body>
</html>
