<?php

namespace App\Models;

use CodeIgniter\Model;

class PermohonanModel extends Model
{
    protected $table = 'permohonans';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    
    protected $allowedFields = [
        'user_id',
        'nomor_perkara',
        'nama_terdakwa',
        'tujuan_permohonan',
        'status',
        'catatan_internal',
        'catatan_pemohon',
        'processed_by',
        'processed_at',
        'pdf_generated_path'
    ];

    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';

    protected $validationRules = [
        'user_id'           => 'required|integer',
        'nomor_perkara'     => 'required|min_length[3]|max_length[100]',
        'nama_terdakwa'     => 'required|min_length[3]|max_length[255]',
        'tujuan_permohonan' => 'required|min_length[10]',
        'status'            => 'in_list[pending,diterima,diproses,selesai,ditolak]',
    ];

    protected $validationMessages = [
        'nomor_perkara' => [
            'required'   => 'Nomor perkara harus diisi',
            'min_length' => 'Nomor perkara minimal 3 karakter',
            'max_length' => 'Nomor perkara maksimal 100 karakter',
        ],
        'nama_terdakwa' => [
            'required'   => 'Nama terdakwa/penggugat harus diisi',
            'min_length' => 'Nama terdakwa/penggugat minimal 3 karakter',
            'max_length' => 'Nama terdakwa/penggugat maksimal 255 karakter',
        ],
        'tujuan_permohonan' => [
            'required'   => 'Tujuan permohonan harus diisi',
            'min_length' => 'Tujuan permohonan minimal 10 karakter',
        ],
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    public function getPermohonanWithUser($permohonanId)
    {
        return $this->select('permohonans.*, users.username, users.email, pemohon_profiles.nama_lengkap, pemohon_profiles.nik, pemohon_profiles.no_telepon')
                    ->join('users', 'users.id = permohonans.user_id')
                    ->join('pemohon_profiles', 'pemohon_profiles.user_id = users.id', 'left')
                    ->where('permohonans.id', $permohonanId)
                    ->first();
    }

    public function getPermohonansByUser($userId)
    {
        return $this->where('user_id', $userId)
                    ->orderBy('created_at', 'DESC')
                    ->findAll();
    }

    public function getAllPermohonansWithUsers()
    {
        return $this->select('permohonans.*, users.username, users.email, pemohon_profiles.nama_lengkap, pemohon_profiles.nik')
                    ->join('users', 'users.id = permohonans.user_id')
                    ->join('pemohon_profiles', 'pemohon_profiles.user_id = users.id', 'left')
                    ->orderBy('permohonans.created_at', 'DESC')
                    ->findAll();
    }

    public function getPermohonansByStatus($status)
    {
        return $this->select('permohonans.*, users.username, users.email, pemohon_profiles.nama_lengkap, pemohon_profiles.nik')
                    ->join('users', 'users.id = permohonans.user_id')
                    ->join('pemohon_profiles', 'pemohon_profiles.user_id = users.id', 'left')
                    ->where('permohonans.status', $status)
                    ->orderBy('permohonans.created_at', 'DESC')
                    ->findAll();
    }

    public function updateStatus($permohonanId, $status, $petugasId, $catatan = null)
    {
        $data = [
            'status'       => $status,
            'processed_by' => $petugasId,
            'processed_at' => date('Y-m-d H:i:s'),
        ];

        if ($catatan) {
            $data['catatan_internal'] = $catatan;
        }

        return $this->update($permohonanId, $data);
    }

    public function getStatistics()
    {
        $stats = [
            'total'    => $this->countAllResults(),
            'pending'  => $this->where('status', 'pending')->countAllResults(),
            'diterima' => $this->where('status', 'diterima')->countAllResults(),
            'diproses' => $this->where('status', 'diproses')->countAllResults(),
            'selesai'  => $this->where('status', 'selesai')->countAllResults(),
            'ditolak'  => $this->where('status', 'ditolak')->countAllResults(),
        ];

        // Get monthly statistics
        $monthlyStats = $this->select('MONTH(created_at) as month, COUNT(*) as count')
                             ->where('YEAR(created_at)', date('Y'))
                             ->groupBy('MONTH(created_at)')
                             ->orderBy('month')
                             ->findAll();

        $stats['monthly'] = array_fill(1, 12, 0);
        foreach ($monthlyStats as $monthly) {
            $stats['monthly'][$monthly['month']] = $monthly['count'];
        }

        return $stats;
    }

    public function searchPermohonans($keyword)
    {
        return $this->select('permohonans.*, users.username, users.email, pemohon_profiles.nama_lengkap, pemohon_profiles.nik')
                    ->join('users', 'users.id = permohonans.user_id')
                    ->join('pemohon_profiles', 'pemohon_profiles.user_id = users.id', 'left')
                    ->groupStart()
                        ->like('permohonans.nomor_perkara', $keyword)
                        ->orLike('permohonans.nama_terdakwa', $keyword)
                        ->orLike('pemohon_profiles.nama_lengkap', $keyword)
                        ->orLike('pemohon_profiles.nik', $keyword)
                    ->groupEnd()
                    ->orderBy('permohonans.created_at', 'DESC')
                    ->findAll();
    }
}
