<?php

namespace App\Models;

use CodeIgniter\Model;

class PermohonanFileModel extends Model
{
    protected $table = 'permohonan_files';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    
    protected $allowedFields = [
        'permohonan_id',
        'jenis_file',
        'nama_file',
        'path_file',
        'ukuran_file',
        'mime_type',
        'uploaded_by'
    ];

    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';

    protected $validationRules = [
        'permohonan_id' => 'required|integer',
        'jenis_file'    => 'required|in_list[ktp,surat_kuasa,pendukung,generated_pdf]',
        'nama_file'     => 'required|max_length[255]',
        'path_file'     => 'required|max_length[255]',
        'ukuran_file'   => 'required|integer',
        'mime_type'     => 'required|max_length[100]',
        'uploaded_by'   => 'required|integer',
    ];

    protected $validationMessages = [
        'permohonan_id' => [
            'required' => 'ID permohonan harus ada',
            'integer'  => 'ID permohonan harus berupa angka',
        ],
        'jenis_file' => [
            'required' => 'Jenis file harus dipilih',
            'in_list'  => 'Jenis file tidak valid',
        ],
        'nama_file' => [
            'required'   => 'Nama file harus ada',
            'max_length' => 'Nama file terlalu panjang',
        ],
        'path_file' => [
            'required'   => 'Path file harus ada',
            'max_length' => 'Path file terlalu panjang',
        ],
        'ukuran_file' => [
            'required' => 'Ukuran file harus ada',
            'integer'  => 'Ukuran file harus berupa angka',
        ],
        'mime_type' => [
            'required'   => 'Tipe MIME file harus ada',
            'max_length' => 'Tipe MIME terlalu panjang',
        ],
        'uploaded_by' => [
            'required' => 'User yang upload harus ada',
            'integer'  => 'User yang upload harus berupa angka',
        ],
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    public function getFilesByPermohonan($permohonanId)
    {
        return $this->where('permohonan_id', $permohonanId)
                    ->orderBy('created_at', 'ASC')
                    ->findAll();
    }

    public function getFilesByJenis($permohonanId, $jenisFile)
    {
        return $this->where('permohonan_id', $permohonanId)
                    ->where('jenis_file', $jenisFile)
                    ->findAll();
    }

    public function getFileWithDetails($fileId)
    {
        return $this->select('permohonan_files.*, users.username as uploader_name')
                    ->join('users', 'users.id = permohonan_files.uploaded_by', 'left')
                    ->where('permohonan_files.id', $fileId)
                    ->first();
    }

    public function deleteFilesByPermohonan($permohonanId)
    {
        // Get all files first to delete physical files
        $files = $this->getFilesByPermohonan($permohonanId);
        
        foreach ($files as $file) {
            $fullPath = WRITEPATH . 'uploads/' . $file['path_file'];
            if (file_exists($fullPath)) {
                unlink($fullPath);
            }
        }

        return $this->where('permohonan_id', $permohonanId)->delete();
    }

    public function deleteFile($fileId)
    {
        $file = $this->find($fileId);
        if ($file) {
            $fullPath = WRITEPATH . 'uploads/' . $file['path_file'];
            if (file_exists($fullPath)) {
                unlink($fullPath);
            }
            return $this->delete($fileId);
        }
        return false;
    }

    public function saveUploadedFile($permohonanId, $jenisFile, $uploadedFile, $uploadedBy)
    {
        // Generate unique filename
        $fileName = $uploadedFile->getName();
        $fileExtension = $uploadedFile->getExtension();
        $newFileName = uniqid() . '_' . time() . '.' . $fileExtension;
        
        // Create directory if not exists
        $uploadPath = WRITEPATH . 'uploads/permohonan/' . $permohonanId . '/';
        if (!is_dir($uploadPath)) {
            mkdir($uploadPath, 0755, true);
        }

        // Move file
        if ($uploadedFile->move($uploadPath, $newFileName)) {
            $fileData = [
                'permohonan_id' => $permohonanId,
                'jenis_file'    => $jenisFile,
                'nama_file'     => $fileName,
                'path_file'     => 'permohonan/' . $permohonanId . '/' . $newFileName,
                'ukuran_file'   => $uploadedFile->getSize(),
                'mime_type'     => $uploadedFile->getMimeType(),
                'uploaded_by'   => $uploadedBy,
            ];

            return $this->insert($fileData);
        }

        return false;
    }

    public function formatFileSize($bytes)
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);

        $bytes /= pow(1024, $pow);

        return round($bytes, 2) . ' ' . $units[$pow];
    }
}
