<?php

namespace App\Models;

use CodeIgniter\Model;

class PemohonProfileModel extends Model
{
    protected $table = 'pemohon_profiles';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    
    protected $allowedFields = [
        'user_id',
        'nama_lengkap',
        'nik',
        'no_telepon',
        'alamat',
        'ktp_path',
        'status_verified',
        'verified_by',
        'verified_at',
        'rejection_reason'
    ];

    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'created_at';
    protected $updatedField = 'updated_at';

    protected $validationRules = [
        'user_id'      => 'required|integer',
        'nama_lengkap' => 'required|min_length[3]|max_length[255]',
        'nik'          => 'required|exact_length[16]|numeric|is_unique[pemohon_profiles.nik,id,{id}]',
        'no_telepon'   => 'required|min_length[10]|max_length[15]',
        'alamat'       => 'required|min_length[10]',
        'status_verified' => 'in_list[pending,verified,rejected]',
    ];

    protected $validationMessages = [
        'nama_lengkap' => [
            'required'   => 'Nama lengkap harus diisi',
            'min_length' => 'Nama lengkap minimal 3 karakter',
            'max_length' => 'Nama lengkap maksimal 255 karakter',
        ],
        'nik' => [
            'required'     => 'NIK harus diisi',
            'exact_length' => 'NIK harus 16 digit',
            'numeric'      => 'NIK harus berupa angka',
            'is_unique'    => 'NIK sudah terdaftar',
        ],
        'no_telepon' => [
            'required'   => 'Nomor telepon harus diisi',
            'min_length' => 'Nomor telepon minimal 10 digit',
            'max_length' => 'Nomor telepon maksimal 15 digit',
        ],
        'alamat' => [
            'required'   => 'Alamat harus diisi',
            'min_length' => 'Alamat minimal 10 karakter',
        ],
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    public function getProfileWithUser($userId)
    {
        return $this->select('pemohon_profiles.*, users.username, users.email, users.is_active')
                    ->join('users', 'users.id = pemohon_profiles.user_id')
                    ->where('pemohon_profiles.user_id', $userId)
                    ->first();
    }

    public function getPendingProfiles()
    {
        return $this->select('pemohon_profiles.*, users.username, users.email, users.created_at as registered_at')
                    ->join('users', 'users.id = pemohon_profiles.user_id')
                    ->where('pemohon_profiles.status_verified', 'pending')
                    ->orderBy('pemohon_profiles.created_at', 'ASC')
                    ->findAll();
    }

    public function verifyProfile($profileId, $petugasId, $status, $reason = null)
    {
        $data = [
            'status_verified' => $status,
            'verified_by'     => $petugasId,
            'verified_at'     => date('Y-m-d H:i:s'),
        ];

        if ($status === 'rejected' && $reason) {
            $data['rejection_reason'] = $reason;
        }

        $result = $this->update($profileId, $data);

        if ($result && $status === 'verified') {
            // Activate user account
            $profile = $this->find($profileId);
            if ($profile) {
                $userModel = new UserModel();
                $userModel->update($profile['user_id'], [
                    'is_active' => true,
                    'email_verified_at' => date('Y-m-d H:i:s')
                ]);
            }
        }

        return $result;
    }

    public function getStatistics()
    {
        $stats = [
            'total' => $this->countAllResults(),
            'pending' => $this->where('status_verified', 'pending')->countAllResults(),
            'verified' => $this->where('status_verified', 'verified')->countAllResults(),
            'rejected' => $this->where('status_verified', 'rejected')->countAllResults(),
        ];

        return $stats;
    }
}
