<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\UserModel;
use App\Models\PemohonProfileModel;

class AuthController extends BaseController
{
    protected $userModel;
    protected $pemohonProfileModel;

    public function __construct()
    {
        $this->userModel = new UserModel();
        $this->pemohonProfileModel = new PemohonProfileModel();
    }

    public function index()
    {
        // Redirect to login if not authenticated
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        // Redirect based on role
        $role = session()->get('role');
        switch ($role) {
            case 'admin':
                return redirect()->to('/admin/dashboard');
            case 'petugas':
                return redirect()->to('/petugas/dashboard');
            case 'pemohon':
                return redirect()->to('/pemohon/dashboard');
            default:
                return redirect()->to('/login');
        }
    }

    public function login()
    {
        if (session()->get('isLoggedIn')) {
            return $this->index();
        }

        $data = [
            'title' => 'Login - Permohonan Salinan Putusan',
            'validation' => \Config\Services::validation()
        ];

        return view('auth/login', $data);
    }

    public function attemptLogin()
    {
        $rules = [
            'email'    => 'required|valid_email',
            'password' => 'required|min_length[6]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $email = $this->request->getPost('email');
        $password = $this->request->getPost('password');

        $user = $this->userModel->verifyLogin($email, $password);

        if (!$user) {
            return redirect()->back()->withInput()->with('error', 'Email atau password salah');
        }

        // For pemohon, check if verified
        if ($user['role'] === 'pemohon') {
            $profile = $this->pemohonProfileModel->getProfileWithUser($user['id']);
            if (!$profile || $profile['status_verified'] !== 'verified') {
                return redirect()->back()->withInput()->with('error', 'Akun Anda belum diverifikasi oleh petugas');
            }
        }

        // Set session
        $sessionData = [
            'user_id'    => $user['id'],
            'username'   => $user['username'],
            'email'      => $user['email'],
            'role'       => $user['role'],
            'isLoggedIn' => true
        ];

        session()->set($sessionData);

        // Redirect based on role
        switch ($user['role']) {
            case 'admin':
                return redirect()->to('/admin/dashboard')->with('success', 'Selamat datang, Admin!');
            case 'petugas':
                return redirect()->to('/petugas/dashboard')->with('success', 'Selamat datang, Petugas!');
            case 'pemohon':
                return redirect()->to('/pemohon/dashboard')->with('success', 'Selamat datang!');
            default:
                return redirect()->to('/login');
        }
    }

    public function register()
    {
        if (session()->get('isLoggedIn')) {
            return $this->index();
        }

        $data = [
            'title' => 'Registrasi Pemohon - Permohonan Salinan Putusan',
            'validation' => \Config\Services::validation()
        ];

        return view('auth/register', $data);
    }

    public function attemptRegister()
    {
        $rules = [
            'username'     => 'required|min_length[3]|max_length[100]|is_unique[users.username]',
            'email'        => 'required|valid_email|is_unique[users.email]',
            'password'     => 'required|min_length[6]',
            'confirm_password' => 'required|matches[password]',
            'nama_lengkap' => 'required|min_length[3]|max_length[255]',
            'nik'          => 'required|exact_length[16]|numeric|is_unique[pemohon_profiles.nik]',
            'no_telepon'   => 'required|min_length[10]|max_length[15]',
            'alamat'       => 'required|min_length[10]',
            'ktp_file'     => 'uploaded[ktp_file]|max_size[ktp_file,5120]|ext_in[ktp_file,pdf,jpg,jpeg,png]',
        ];

        $messages = [
            'username' => [
                'required'    => 'Username harus diisi',
                'min_length'  => 'Username minimal 3 karakter',
                'max_length'  => 'Username maksimal 100 karakter',
                'is_unique'   => 'Username sudah digunakan',
            ],
            'email' => [
                'required'    => 'Email harus diisi',
                'valid_email' => 'Email tidak valid',
                'is_unique'   => 'Email sudah terdaftar',
            ],
            'password' => [
                'required'   => 'Password harus diisi',
                'min_length' => 'Password minimal 6 karakter',
            ],
            'confirm_password' => [
                'required' => 'Konfirmasi password harus diisi',
                'matches'  => 'Konfirmasi password tidak sama',
            ],
            'nama_lengkap' => [
                'required'   => 'Nama lengkap harus diisi',
                'min_length' => 'Nama lengkap minimal 3 karakter',
                'max_length' => 'Nama lengkap maksimal 255 karakter',
            ],
            'nik' => [
                'required'     => 'NIK harus diisi',
                'exact_length' => 'NIK harus 16 digit',
                'numeric'      => 'NIK harus berupa angka',
                'is_unique'    => 'NIK sudah terdaftar',
            ],
            'no_telepon' => [
                'required'   => 'Nomor telepon harus diisi',
                'min_length' => 'Nomor telepon minimal 10 digit',
                'max_length' => 'Nomor telepon maksimal 15 digit',
            ],
            'alamat' => [
                'required'   => 'Alamat harus diisi',
                'min_length' => 'Alamat minimal 10 karakter',
            ],
            'ktp_file' => [
                'uploaded' => 'File KTP harus diupload',
                'max_size' => 'Ukuran file KTP maksimal 5MB',
                'ext_in'   => 'File KTP harus berformat PDF, JPG, JPEG, atau PNG',
            ],
        ];

        if (!$this->validate($rules, $messages)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        // Start transaction
        $db = \Config\Database::connect();
        $db->transStart();

        try {
            // Insert user
            $userData = [
                'username' => $this->request->getPost('username'),
                'email'    => $this->request->getPost('email'),
                'password' => $this->request->getPost('password'),
                'role'     => 'pemohon',
                'is_active' => false, // Will be activated after verification
            ];

            $userId = $this->userModel->insert($userData);

            if (!$userId) {
                throw new \Exception('Gagal menyimpan data user');
            }

            // Handle KTP file upload
            $ktpFile = $this->request->getFile('ktp_file');
            $ktpPath = null;

            if ($ktpFile && $ktpFile->isValid() && !$ktpFile->hasMoved()) {
                $newFileName = 'ktp_' . $userId . '_' . time() . '.' . $ktpFile->getExtension();
                $uploadPath = WRITEPATH . 'uploads/ktp/';
                
                if (!is_dir($uploadPath)) {
                    mkdir($uploadPath, 0755, true);
                }

                if ($ktpFile->move($uploadPath, $newFileName)) {
                    $ktpPath = 'ktp/' . $newFileName;
                }
            }

            // Insert pemohon profile
            $profileData = [
                'user_id'      => $userId,
                'nama_lengkap' => $this->request->getPost('nama_lengkap'),
                'nik'          => $this->request->getPost('nik'),
                'no_telepon'   => $this->request->getPost('no_telepon'),
                'alamat'       => $this->request->getPost('alamat'),
                'ktp_path'     => $ktpPath,
                'status_verified' => 'pending',
            ];

            $profileId = $this->pemohonProfileModel->insert($profileData);

            if (!$profileId) {
                throw new \Exception('Gagal menyimpan data profile');
            }

            $db->transComplete();

            if ($db->transStatus() === false) {
                throw new \Exception('Transaksi database gagal');
            }

            // Send email notification (implement later)
            // $this->sendRegistrationEmail($userData['email'], $userData['username']);

            return redirect()->to('/login')->with('success', 'Registrasi berhasil! Silakan tunggu verifikasi dari petugas.');

        } catch (\Exception $e) {
            $db->transRollback();
            
            // Delete uploaded file if exists
            if ($ktpPath) {
                $fullPath = WRITEPATH . 'uploads/' . $ktpPath;
                if (file_exists($fullPath)) {
                    unlink($fullPath);
                }
            }

            return redirect()->back()->withInput()->with('error', 'Registrasi gagal: ' . $e->getMessage());
        }
    }

    public function logout()
    {
        session()->destroy();
        return redirect()->to('/login')->with('success', 'Anda telah logout');
    }

    private function sendRegistrationEmail($email, $username)
    {
        // Implementation will be added later
        $emailService = \Config\Services::email();
        
        $emailService->setTo($email);
        $emailService->setFrom('noreply@pn-watampone.go.id', 'PN Watampone');
        $emailService->setSubject('Registrasi Berhasil - Menunggu Verifikasi');
        
        $message = "Halo {$username},\n\n";
        $message .= "Registrasi Anda untuk aplikasi Permohonan Salinan Putusan telah berhasil.\n";
        $message .= "Akun Anda saat ini sedang menunggu verifikasi oleh petugas.\n\n";
        $message .= "Anda akan mendapat notifikasi email ketika akun telah diverifikasi.\n\n";
        $message .= "Terima kasih,\n";
        $message .= "Pengadilan Negeri Watampone";
        
        $emailService->setMessage($message);
        
        return $emailService->send();
    }
}
