<?php
require_once 'config/config.php';
require_once 'includes/functions.php';

// Require login and admin role
$auth->requireAdmin();

$page_title = 'Persetujuan Izin';
$current_user = $auth->getCurrentUser();

// Get filters
$status_filter = $_GET['status'] ?? 'pending';
$user_filter = $_GET['user'] ?? '';
$page = max(1, intval($_GET['page'] ?? 1));

// Handle approval/rejection
if ($_POST && isset($_POST['action'])) {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        redirect('persetujuan.php', 'Token keamanan tidak valid', 'error');
    }
    
    $permission_id = intval($_POST['permission_id']);
    $action = $_POST['action'];
    $notes = sanitize($_POST['notes'] ?? '');
    
    if ($action === 'approve') {
        $result = $permissionManager->updatePermissionStatus($permission_id, 'approved', $current_user['id'], $notes);
    } elseif ($action === 'reject') {
        $result = $permissionManager->updatePermissionStatus($permission_id, 'rejected', $current_user['id'], $notes);
    }
    
    if (isset($result)) {
        redirect('persetujuan.php', $result['message'], $result['success'] ? 'success' : 'error');
    }
}

// Build filters
$filters = [];
if ($status_filter && $status_filter !== 'all') $filters['status'] = $status_filter;
if ($user_filter) $filters['user_id'] = $user_filter;

// Get permissions
$result = $permissionManager->getAllPermissions($page, 10, $filters);
$permissions = $result['success'] ? $result['data'] : [];
$total_pages = $result['success'] ? $result['total_pages'] : 1;

// Get users for filter
try {
    $stmt = $pdo->prepare("SELECT id, name, nip FROM users WHERE role = 'user' ORDER BY name");
    $stmt->execute();
    $users = $stmt->fetchAll();
} catch(PDOException $e) {
    $users = [];
}

$csrf_token = generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - SISKIM PN Watampone</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/feather-icons/dist/feather.min.js"></script>
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        .sidebar { transition: all 0.3s; }
        .nav-item.active { background-color: #1e3a8a; color: white; }
        .nav-item.active svg { color: white; }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen overflow-hidden">
        <!-- Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <div class="flex-1 overflow-auto">
            <!-- Header -->
            <?php include 'includes/header.php'; ?>

            <!-- Content -->
            <main class="p-6">
                <!-- Breadcrumb -->
                <?php 
                generateBreadcrumb([
                    ['title' => 'Dashboard', 'url' => 'dashboard.php', 'icon' => 'home'],
                    ['title' => 'Persetujuan Izin', 'url' => '']
                ]); 
                ?>
                
                <!-- Flash Messages -->
                <?php showFlashMessage(); ?>
                
                <div class="mb-6">
                    <h1 class="text-2xl font-bold text-gray-800">Persetujuan Pengajuan Izin</h1>
                    <p class="text-gray-600 mt-1">Kelola persetujuan pengajuan izin dari pegawai</p>
                </div>

                <!-- Filters -->
                <div class="bg-white rounded-lg shadow p-6 mb-6">
                    <h3 class="text-lg font-semibold mb-4">Filter Pencarian</h3>
                    <form method="GET" class="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                            <select name="status" class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                                <option value="pending" <?php echo ($status_filter === 'pending') ? 'selected' : ''; ?>>Menunggu Persetujuan</option>
                                <option value="approved" <?php echo ($status_filter === 'approved') ? 'selected' : ''; ?>>Disetujui</option>
                                <option value="rejected" <?php echo ($status_filter === 'rejected') ? 'selected' : ''; ?>>Ditolak</option>
                                <option value="all" <?php echo ($status_filter === 'all') ? 'selected' : ''; ?>>Semua Status</option>
                            </select>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Pegawai</label>
                            <select name="user" class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                                <option value="">Semua Pegawai</option>
                                <?php foreach ($users as $user): ?>
                                <option value="<?php echo $user['id']; ?>" <?php echo ($user_filter == $user['id']) ? 'selected' : ''; ?>>
                                    <?php echo escape($user['name']) . ' (' . escape($user['nip']) . ')'; ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="flex items-end">
                            <button type="submit" class="w-full bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 flex items-center justify-center">
                                <i data-feather="filter" class="w-4 h-4 mr-2"></i>
                                Filter
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Data Table -->
                <div class="bg-white rounded-lg shadow overflow-hidden">
                    <div class="overflow-x-auto">
                        <?php if (!empty($permissions)): ?>
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Pegawai</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Jenis Izin</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tanggal & Waktu</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Keperluan</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Aksi</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php foreach ($permissions as $permission): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="flex items-center">
                                            <div class="flex-shrink-0 h-10 w-10">
                                                <div class="h-10 w-10 rounded-full bg-gray-300 flex items-center justify-center">
                                                    <i data-feather="user" class="h-5 w-5 text-gray-600"></i>
                                                </div>
                                            </div>
                                            <div class="ml-4">
                                                <div class="text-sm font-medium text-gray-900"><?php echo escape($permission['user_name']); ?></div>
                                                <div class="text-sm text-gray-500"><?php echo escape($permission['nip']); ?></div>
                                                <div class="text-xs text-gray-400"><?php echo escape($permission['position']); ?></div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?php echo escape($permission['permission_type_name']); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                        <div><?php echo formatTanggalIndonesia($permission['request_date']); ?></div>
                                        <div class="text-xs">
                                            <?php echo formatWaktu($permission['start_time']) . ' - ' . formatWaktu($permission['end_time']); ?>
                                        </div>
                                        <div class="text-xs text-gray-400">
                                            (<?php echo hitungDurasi($permission['start_time'], $permission['end_time']); ?>)
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900 max-w-xs">
                                        <div class="truncate" title="<?php echo escape($permission['purpose']); ?>">
                                            <?php echo escape(substr($permission['purpose'], 0, 50)) . (strlen($permission['purpose']) > 50 ? '...' : ''); ?>
                                        </div>
                                        <div class="text-xs text-gray-500 mt-1">
                                            <strong>Tujuan:</strong> <?php echo escape(substr($permission['destination'], 0, 30)) . (strlen($permission['destination']) > 30 ? '...' : ''); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <?php echo getStatusBadge($permission['status']); ?>
                                        <?php if ($permission['approved_by_name']): ?>
                                        <div class="text-xs text-gray-500 mt-1">
                                            oleh: <?php echo escape($permission['approved_by_name']); ?>
                                        </div>
                                        <?php endif; ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                                        <?php if ($permission['status'] === 'pending'): ?>
                                        <button onclick="approveModal(<?php echo $permission['id']; ?>, '<?php echo escape($permission['user_name']); ?>')" class="text-green-600 hover:text-green-900">
                                            <i data-feather="check-circle" class="w-4 h-4 inline"></i> Setujui
                                        </button>
                                        <button onclick="rejectModal(<?php echo $permission['id']; ?>, '<?php echo escape($permission['user_name']); ?>')" class="text-red-600 hover:text-red-900">
                                            <i data-feather="x-circle" class="w-4 h-4 inline"></i> Tolak
                                        </button>
                                        <?php endif; ?>
                                        <a href="detail.php?id=<?php echo $permission['id']; ?>" class="text-blue-600 hover:text-blue-900">
                                            <i data-feather="eye" class="w-4 h-4 inline"></i> Detail
                                        </a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        
                        <!-- Pagination -->
                        <?php echo generatePagination($page, $total_pages, 'persetujuan.php'); ?>
                        
                        <?php else: ?>
                        <div class="p-12 text-center">
                            <i data-feather="check-circle" class="w-16 h-16 mx-auto text-gray-300 mb-4"></i>
                            <h3 class="text-lg font-medium text-gray-900 mb-2">Tidak ada pengajuan</h3>
                            <p class="text-gray-500">Saat ini tidak ada pengajuan izin yang perlu disetujui.</p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Approval Modal -->
    <div id="approveModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <form method="POST">
                <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                <input type="hidden" name="permission_id" id="approve_permission_id">
                <input type="hidden" name="action" value="approve">
                
                <div class="mt-3 text-center">
                    <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-green-100">
                        <i data-feather="check-circle" class="h-6 w-6 text-green-600"></i>
                    </div>
                    <h3 class="text-lg font-medium text-gray-900 mt-2">Setujui Pengajuan</h3>
                    <div class="mt-2 px-7 py-3">
                        <p class="text-sm text-gray-500" id="approve_message"></p>
                        <textarea name="notes" class="mt-3 w-full border border-gray-300 rounded-lg p-2 text-sm" rows="3" placeholder="Catatan persetujuan (opsional)"></textarea>
                    </div>
                    <div class="items-center px-4 py-3">
                        <button type="submit" class="px-4 py-2 bg-green-500 text-white text-base font-medium rounded-md w-24 mr-2 hover:bg-green-600">
                            Setujui
                        </button>
                        <button type="button" onclick="closeModal('approveModal')" class="px-4 py-2 bg-gray-300 text-gray-800 text-base font-medium rounded-md w-24 hover:bg-gray-400">
                            Batal
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Rejection Modal -->
    <div id="rejectModal" class="hidden fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <form method="POST">
                <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                <input type="hidden" name="permission_id" id="reject_permission_id">
                <input type="hidden" name="action" value="reject">
                
                <div class="mt-3 text-center">
                    <div class="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-red-100">
                        <i data-feather="x-circle" class="h-6 w-6 text-red-600"></i>
                    </div>
                    <h3 class="text-lg font-medium text-gray-900 mt-2">Tolak Pengajuan</h3>
                    <div class="mt-2 px-7 py-3">
                        <p class="text-sm text-gray-500" id="reject_message"></p>
                        <textarea name="notes" class="mt-3 w-full border border-gray-300 rounded-lg p-2 text-sm" rows="3" placeholder="Alasan penolakan *" required></textarea>
                    </div>
                    <div class="items-center px-4 py-3">
                        <button type="submit" class="px-4 py-2 bg-red-500 text-white text-base font-medium rounded-md w-24 mr-2 hover:bg-red-600">
                            Tolak
                        </button>
                        <button type="button" onclick="closeModal('rejectModal')" class="px-4 py-2 bg-gray-300 text-gray-800 text-base font-medium rounded-md w-24 hover:bg-gray-400">
                            Batal
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <script>
        feather.replace();
        
        function approveModal(id, userName) {
            document.getElementById('approve_permission_id').value = id;
            document.getElementById('approve_message').textContent = `Apakah Anda yakin ingin menyetujui pengajuan izin dari ${userName}?`;
            document.getElementById('approveModal').classList.remove('hidden');
        }
        
        function rejectModal(id, userName) {
            document.getElementById('reject_permission_id').value = id;
            document.getElementById('reject_message').textContent = `Apakah Anda yakin ingin menolak pengajuan izin dari ${userName}?`;
            document.getElementById('rejectModal').classList.remove('hidden');
        }
        
        function closeModal(modalId) {
            document.getElementById(modalId).classList.add('hidden');
        }
        
        // Mobile menu toggle
        function toggleSidebar() {
            document.querySelector('.sidebar').classList.toggle('hidden');
        }
    </script>
</body>
</html>