<?php
require_once 'config/config.php';
require_once 'includes/functions.php';

// Require login
$auth->requireLogin();

$page_title = 'Form Pengajuan Izin';
$current_user = $auth->getCurrentUser();

// Get permission types
$permission_types = $permissionManager->getPermissionTypes();

$error_message = '';
$success_message = '';

// Handle form submission
if ($_POST) {
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $error_message = 'Token keamanan tidak valid';
    } else {
        // Sanitize input
        $data = [
            'user_id' => $current_user['id'],
            'permission_type_id' => sanitize($_POST['permission_type_id'] ?? ''),
            'request_date' => sanitize($_POST['request_date'] ?? ''),
            'start_time' => sanitize($_POST['start_time'] ?? ''),
            'end_time' => sanitize($_POST['end_time'] ?? ''),
            'purpose' => sanitize($_POST['purpose'] ?? ''),
            'destination' => sanitize($_POST['destination'] ?? ''),
            'notes' => sanitize($_POST['notes'] ?? '')
        ];
        
        // Handle file upload
        if (isset($_FILES['document']) && $_FILES['document']['error'] == UPLOAD_ERR_OK) {
            $upload_result = uploadFile($_FILES['document']);
            if ($upload_result['success']) {
                $data['document_path'] = $upload_result['filename'];
            } else {
                $error_message = $upload_result['message'];
            }
        }
        
        // Create permission if no upload error
        if (empty($error_message)) {
            $result = $permissionManager->createPermission($data);
            
            if ($result['success']) {
                redirect('dashboard.php', 'Pengajuan izin berhasil dibuat! Menunggu persetujuan atasan.');
            } else {
                $error_message = $result['message'];
            }
        }
    }
}

// Get selected type from URL parameter
$selected_type = $_GET['type'] ?? '';

$csrf_token = generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - SISKIM PN Watampone</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://unpkg.com/aos@2.3.1/dist/aos.css" rel="stylesheet">
    <script src="https://unpkg.com/aos@2.3.1/dist/aos.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/feather-icons/dist/feather.min.js"></script>
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        .sidebar {
            transition: all 0.3s;
        }
        .nav-item.active {
            background-color: #1e3a8a;
            color: white;
        }
        .nav-item.active svg {
            color: white;
        }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen overflow-hidden">
        <!-- Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <div class="flex-1 overflow-auto">
            <!-- Header -->
            <?php include 'includes/header.php'; ?>

            <!-- Content -->
            <main class="p-6">
                <!-- Breadcrumb -->
                <?php 
                generateBreadcrumb([
                    ['title' => 'Dashboard', 'url' => 'dashboard.php', 'icon' => 'home'],
                    ['title' => 'Pengajuan Izin', 'url' => '']
                ]); 
                ?>
                
                <!-- Flash Messages -->
                <?php showFlashMessage(); ?>
                
                <?php if ($error_message): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4" role="alert">
                    <div class="flex items-center">
                        <i data-feather="alert-circle" class="w-5 h-5 mr-2"></i>
                        <span><?php echo escape($error_message); ?></span>
                    </div>
                </div>
                <?php endif; ?>

                <div class="mb-6 flex justify-between items-center">
                    <div>
                        <h1 class="text-2xl font-bold text-gray-800">Form Pengajuan Izin</h1>
                        <p class="text-gray-600 mt-1">Isi form berikut untuk mengajukan izin keluar/masuk kantor</p>
                    </div>
                    <a href="dashboard.php" class="text-sm text-blue-600 hover:underline flex items-center">
                        <i data-feather="arrow-left" class="w-4 h-4 mr-1"></i> Kembali ke Dashboard
                    </a>
                </div>

                <div class="bg-white rounded-lg shadow overflow-hidden">
                    <div class="p-6 border-b border-gray-200">
                        <h2 class="text-lg font-semibold">Informasi Pengajuan</h2>
                        <p class="text-sm text-gray-500 mt-1">Pastikan semua informasi yang Anda masukkan sudah benar</p>
                    </div>
                    
                    <div class="p-6">
                        <form method="POST" action="" enctype="multipart/form-data">
                            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                            
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <label for="permission_type_id" class="block text-sm font-medium text-gray-700 mb-1">
                                        Jenis Izin <span class="text-red-500">*</span>
                                    </label>
                                    <select 
                                        id="permission_type_id" 
                                        name="permission_type_id" 
                                        class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3"
                                        required
                                    >
                                        <option value="">Pilih jenis izin</option>
                                        <?php foreach ($permission_types as $type): ?>
                                        <option value="<?php echo $type['id']; ?>" <?php echo ($selected_type == $type['id']) ? 'selected' : ''; ?>>
                                            <?php echo escape($type['name']); ?>
                                        </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                
                                <div>
                                    <label for="request_date" class="block text-sm font-medium text-gray-700 mb-1">
                                        Tanggal Pengajuan <span class="text-red-500">*</span>
                                    </label>
                                    <div class="relative">
                                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <i data-feather="calendar" class="h-5 w-5 text-gray-400"></i>
                                        </div>
                                        <input 
                                            type="date" 
                                            id="request_date" 
                                            name="request_date" 
                                            class="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                                            min="<?php echo date('Y-m-d'); ?>"
                                            value="<?php echo escape($_POST['request_date'] ?? date('Y-m-d')); ?>"
                                            required
                                        >
                                    </div>
                                </div>
                                
                                <div>
                                    <label for="start_time" class="block text-sm font-medium text-gray-700 mb-1">
                                        Waktu Keluar <span class="text-red-500">*</span>
                                    </label>
                                    <div class="relative">
                                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <i data-feather="clock" class="h-5 w-5 text-gray-400"></i>
                                        </div>
                                        <input 
                                            type="time" 
                                            id="start_time" 
                                            name="start_time" 
                                            class="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                                            value="<?php echo escape($_POST['start_time'] ?? ''); ?>"
                                            required
                                        >
                                    </div>
                                </div>
                                
                                <div>
                                    <label for="end_time" class="block text-sm font-medium text-gray-700 mb-1">
                                        Waktu Kembali <span class="text-red-500">*</span>
                                    </label>
                                    <div class="relative">
                                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <i data-feather="clock" class="h-5 w-5 text-gray-400"></i>
                                        </div>
                                        <input 
                                            type="time" 
                                            id="end_time" 
                                            name="end_time" 
                                            class="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                                            value="<?php echo escape($_POST['end_time'] ?? ''); ?>"
                                            required
                                        >
                                    </div>
                                    <p class="text-xs text-gray-500 mt-1">Estimasi waktu kembali ke kantor</p>
                                </div>
                            </div>

                            <div class="mt-6">
                                <label for="purpose" class="block text-sm font-medium text-gray-700 mb-1">
                                    Tujuan/Keperluan <span class="text-red-500">*</span>
                                </label>
                                <textarea 
                                    id="purpose" 
                                    name="purpose" 
                                    rows="3" 
                                    class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3" 
                                    placeholder="Jelaskan tujuan/keperluan izin Anda dengan detail"
                                    required
                                ><?php echo escape($_POST['purpose'] ?? ''); ?></textarea>
                            </div>

                            <div class="mt-6">
                                <label for="destination" class="block text-sm font-medium text-gray-700 mb-1">
                                    Alamat Tujuan <span class="text-red-500">*</span>
                                </label>
                                <textarea 
                                    id="destination" 
                                    name="destination" 
                                    rows="2" 
                                    class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3" 
                                    placeholder="Masukkan alamat lengkap tujuan"
                                    required
                                ><?php echo escape($_POST['destination'] ?? ''); ?></textarea>
                            </div>
                            
                            <div class="mt-6">
                                <label for="notes" class="block text-sm font-medium text-gray-700 mb-1">
                                    Catatan Tambahan
                                </label>
                                <textarea 
                                    id="notes" 
                                    name="notes" 
                                    rows="2" 
                                    class="block w-full border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 py-2 px-3" 
                                    placeholder="Catatan atau informasi tambahan (opsional)"
                                ><?php echo escape($_POST['notes'] ?? ''); ?></textarea>
                            </div>

                            <div class="mt-6">
                                <label for="document" class="block text-sm font-medium text-gray-700 mb-1">
                                    Dokumen Pendukung (Opsional)
                                </label>
                                <div class="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-lg">
                                    <div class="space-y-1 text-center">
                                        <i data-feather="upload" class="mx-auto h-12 w-12 text-gray-400"></i>
                                        <div class="flex text-sm text-gray-600">
                                            <label for="document" class="relative cursor-pointer bg-white rounded-md font-medium text-blue-600 hover:text-blue-500 focus-within:outline-none">
                                                <span>Upload file</span>
                                                <input 
                                                    id="document" 
                                                    name="document" 
                                                    type="file" 
                                                    class="sr-only"
                                                    accept=".pdf,.jpg,.jpeg,.png"
                                                    onchange="updateFileName(this)"
                                                >
                                            </label>
                                            <p class="pl-1">atau drag and drop</p>
                                        </div>
                                        <p class="text-xs text-gray-500">PDF, JPG, PNG maksimal 5MB</p>
                                        <p id="file-name" class="text-xs text-gray-700 font-medium"></p>
                                    </div>
                                </div>
                            </div>

                            <div class="mt-8 flex justify-end space-x-3">
                                <a href="dashboard.php" class="inline-flex justify-center py-2 px-4 border border-gray-300 shadow-sm text-sm font-medium rounded-lg text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                    Batal
                                </a>
                                <button type="submit" class="inline-flex justify-center py-2 px-4 border border-transparent shadow-sm text-sm font-medium rounded-lg text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500">
                                    <i data-feather="send" class="w-4 h-4 mr-2"></i>
                                    Ajukan Izin
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script>
        AOS.init();
        feather.replace();
        
        // Update file name display
        function updateFileName(input) {
            const fileNameEl = document.getElementById('file-name');
            if (input.files && input.files[0]) {
                fileNameEl.textContent = 'File terpilih: ' + input.files[0].name;
            } else {
                fileNameEl.textContent = '';
            }
        }
        
        // Validate time selection
        document.getElementById('start_time').addEventListener('change', function() {
            const startTime = this.value;
            const endTimeInput = document.getElementById('end_time');
            
            if (startTime && endTimeInput.value && endTimeInput.value <= startTime) {
                alert('Waktu kembali harus setelah waktu keluar');
                endTimeInput.value = '';
            }
        });
        
        document.getElementById('end_time').addEventListener('change', function() {
            const endTime = this.value;
            const startTimeInput = document.getElementById('start_time');
            
            if (startTimeInput.value && endTime <= startTimeInput.value) {
                alert('Waktu kembali harus setelah waktu keluar');
                this.value = '';
            }
        });
        
        // Mobile menu toggle
        function toggleSidebar() {
            document.querySelector('.sidebar').classList.toggle('hidden');
        }
    </script>
</body>
</html>