<?php
require_once 'config/config.php';
require_once 'includes/functions.php';

// Require admin role
$auth->requireAdmin();

$page_title = 'Laporan';
$current_user = $auth->getCurrentUser();

// Get date range from filters
$date_from = $_GET['date_from'] ?? date('Y-m-01'); // First day of current month
$date_to = $_GET['date_to'] ?? date('Y-m-t');       // Last day of current month
$user_filter = $_GET['user'] ?? '';
$status_filter = $_GET['status'] ?? '';
$export = $_GET['export'] ?? '';

// Build filters
$filters = [];
if ($date_from) $filters['date_from'] = $date_from;
if ($date_to) $filters['date_to'] = $date_to;
if ($user_filter) $filters['user_id'] = $user_filter;
if ($status_filter) $filters['status'] = $status_filter;

// Get data for report
$result = $permissionManager->getAllPermissions(1, 1000, $filters); // Get all data for report
$permissions = $result['success'] ? $result['data'] : [];

// Get users for filter
try {
    $stmt = $pdo->prepare("SELECT id, name, nip FROM users WHERE role = 'user' ORDER BY name");
    $stmt->execute();
    $users = $stmt->fetchAll();
} catch(PDOException $e) {
    $users = [];
}

// Calculate statistics
$stats = [
    'total' => count($permissions),
    'approved' => count(array_filter($permissions, fn($p) => $p['status'] === 'approved')),
    'pending' => count(array_filter($permissions, fn($p) => $p['status'] === 'pending')),
    'rejected' => count(array_filter($permissions, fn($p) => $p['status'] === 'rejected'))
];

// Group by permission type
$by_type = [];
foreach ($permissions as $permission) {
    $type = $permission['permission_type_name'];
    if (!isset($by_type[$type])) {
        $by_type[$type] = 0;
    }
    $by_type[$type]++;
}

// Handle CSV export
if ($export === 'csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="laporan_izin_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // CSV Headers
    fputcsv($output, [
        'No',
        'NIP',
        'Nama',
        'Jabatan',
        'Jenis Izin',
        'Tanggal',
        'Waktu Keluar',
        'Waktu Kembali',
        'Durasi',
        'Keperluan',
        'Tujuan',
        'Status',
        'Disetujui Oleh',
        'Tanggal Disetujui'
    ]);
    
    // CSV Data
    $no = 1;
    foreach ($permissions as $permission) {
        fputcsv($output, [
            $no++,
            $permission['nip'],
            $permission['user_name'],
            $permission['position'],
            $permission['permission_type_name'],
            formatTanggalIndonesia($permission['request_date']),
            formatWaktu($permission['start_time']),
            formatWaktu($permission['end_time']),
            hitungDurasi($permission['start_time'], $permission['end_time']),
            $permission['purpose'],
            $permission['destination'],
            ucfirst($permission['status']),
            $permission['approved_by_name'] ?: '-',
            $permission['approved_at'] ? formatTanggalIndonesia($permission['approved_at'], 'd F Y H:i') : '-'
        ]);
    }
    
    fclose($output);
    exit();
}
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - SISKIM PN Watampone</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/feather-icons/dist/feather.min.js"></script>
    <script src="https://unpkg.com/feather-icons"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        .sidebar { transition: all 0.3s; }
        .nav-item.active { background-color: #1e3a8a; color: white; }
        .nav-item.active svg { color: white; }
        @media print {
            .no-print { display: none; }
            .sidebar { display: none; }
            .print-full-width { width: 100% !important; }
        }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen overflow-hidden">
        <!-- Sidebar -->
        <div class="no-print">
            <?php include 'includes/sidebar.php'; ?>
        </div>

        <!-- Main Content -->
        <div class="flex-1 overflow-auto print-full-width">
            <!-- Header -->
            <div class="no-print">
                <?php include 'includes/header.php'; ?>
            </div>

            <!-- Content -->
            <main class="p-6">
                <!-- Breadcrumb -->
                <div class="no-print">
                    <?php 
                    generateBreadcrumb([
                        ['title' => 'Dashboard', 'url' => 'dashboard.php', 'icon' => 'home'],
                        ['title' => 'Laporan', 'url' => '']
                    ]); 
                    ?>
                </div>
                
                <!-- Flash Messages -->
                <div class="no-print">
                    <?php showFlashMessage(); ?>
                </div>
                
                <div class="mb-6 flex justify-between items-center">
                    <div>
                        <h1 class="text-2xl font-bold text-gray-800">Laporan Pengajuan Izin</h1>
                        <p class="text-gray-600 mt-1">Analisis dan statistik pengajuan izin pegawai</p>
                    </div>
                    <div class="no-print flex space-x-2">
                        <button onclick="window.print()" class="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 flex items-center">
                            <i data-feather="printer" class="w-4 h-4 mr-2"></i>
                            Cetak
                        </button>
                        <a href="?<?php echo http_build_query(array_merge($_GET, ['export' => 'csv'])); ?>" class="bg-green-600 text-white px-4 py-2 rounded-lg hover:bg-green-700 flex items-center">
                            <i data-feather="download" class="w-4 h-4 mr-2"></i>
                            Export CSV
                        </a>
                    </div>
                </div>

                <!-- Filters -->
                <div class="no-print bg-white rounded-lg shadow p-6 mb-6">
                    <h3 class="text-lg font-semibold mb-4">Filter Laporan</h3>
                    <form method="GET" class="grid grid-cols-1 md:grid-cols-5 gap-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Tanggal Dari</label>
                            <input type="date" name="date_from" value="<?php echo $date_from; ?>" class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Tanggal Sampai</label>
                            <input type="date" name="date_to" value="<?php echo $date_to; ?>" class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Pegawai</label>
                            <select name="user" class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                                <option value="">Semua Pegawai</option>
                                <?php foreach ($users as $user): ?>
                                <option value="<?php echo $user['id']; ?>" <?php echo ($user_filter == $user['id']) ? 'selected' : ''; ?>>
                                    <?php echo escape($user['name']); ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                            <select name="status" class="block w-full border border-gray-300 rounded-lg py-2 px-3">
                                <option value="">Semua Status</option>
                                <option value="pending" <?php echo ($status_filter === 'pending') ? 'selected' : ''; ?>>Menunggu</option>
                                <option value="approved" <?php echo ($status_filter === 'approved') ? 'selected' : ''; ?>>Disetujui</option>
                                <option value="rejected" <?php echo ($status_filter === 'rejected') ? 'selected' : ''; ?>>Ditolak</option>
                            </select>
                        </div>
                        <div class="flex items-end">
                            <button type="submit" class="w-full bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center justify-center">
                                <i data-feather="search" class="w-4 h-4 mr-2"></i>
                                Filter
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Report Header (for print) -->
                <div class="hidden print:block mb-6 text-center">
                    <h1 class="text-2xl font-bold">LAPORAN PENGAJUAN IZIN</h1>
                    <h2 class="text-lg">Pengadilan Negeri Watampone</h2>
                    <p class="text-sm mt-2">
                        Periode: <?php echo formatTanggalIndonesia($date_from) . ' - ' . formatTanggalIndonesia($date_to); ?>
                    </p>
                    <p class="text-sm">Dicetak pada: <?php echo formatTanggalIndonesia(date('Y-m-d'), 'd F Y H:i'); ?></p>
                </div>

                <!-- Statistics Cards -->
                <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-6">
                    <div class="bg-white rounded-lg shadow p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm text-gray-500">Total Pengajuan</p>
                                <h3 class="text-2xl font-bold mt-1"><?php echo $stats['total']; ?></h3>
                            </div>
                            <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                                <i data-feather="file-text" class="w-6 h-6"></i>
                            </div>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm text-gray-500">Disetujui</p>
                                <h3 class="text-2xl font-bold mt-1 text-green-600"><?php echo $stats['approved']; ?></h3>
                            </div>
                            <div class="p-3 rounded-full bg-green-100 text-green-600">
                                <i data-feather="check-circle" class="w-6 h-6"></i>
                            </div>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm text-gray-500">Menunggu</p>
                                <h3 class="text-2xl font-bold mt-1 text-yellow-600"><?php echo $stats['pending']; ?></h3>
                            </div>
                            <div class="p-3 rounded-full bg-yellow-100 text-yellow-600">
                                <i data-feather="clock" class="w-6 h-6"></i>
                            </div>
                        </div>
                    </div>
                    <div class="bg-white rounded-lg shadow p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <p class="text-sm text-gray-500">Ditolak</p>
                                <h3 class="text-2xl font-bold mt-1 text-red-600"><?php echo $stats['rejected']; ?></h3>
                            </div>
                            <div class="p-3 rounded-full bg-red-100 text-red-600">
                                <i data-feather="x-circle" class="w-6 h-6"></i>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Charts -->
                <div class="no-print grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
                    <!-- Status Chart -->
                    <div class="bg-white rounded-lg shadow p-6">
                        <h3 class="text-lg font-semibold mb-4">Distribusi Status</h3>
                        <canvas id="statusChart" width="400" height="200"></canvas>
                    </div>
                    
                    <!-- Type Chart -->
                    <div class="bg-white rounded-lg shadow p-6">
                        <h3 class="text-lg font-semibold mb-4">Jenis Izin</h3>
                        <canvas id="typeChart" width="400" height="200"></canvas>
                    </div>
                </div>

                <!-- Data Table -->
                <div class="bg-white rounded-lg shadow overflow-hidden">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-semibold">Detail Pengajuan</h2>
                    </div>
                    <div class="overflow-x-auto">
                        <?php if (!empty($permissions)): ?>
                        <table class="min-w-full divide-y divide-gray-200 text-sm">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">No</th>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Pegawai</th>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Jenis</th>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tanggal</th>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Waktu</th>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Keperluan</th>
                                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php foreach ($permissions as $index => $permission): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo $index + 1; ?>
                                    </td>
                                    <td class="px-4 py-3 whitespace-nowrap">
                                        <div class="text-sm font-medium text-gray-900"><?php echo escape($permission['user_name']); ?></div>
                                        <div class="text-xs text-gray-500"><?php echo escape($permission['nip']); ?></div>
                                    </td>
                                    <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-900">
                                        <?php echo escape($permission['permission_type_name']); ?>
                                    </td>
                                    <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo formatTanggalIndonesia($permission['request_date']); ?>
                                    </td>
                                    <td class="px-4 py-3 whitespace-nowrap text-sm text-gray-500">
                                        <?php echo formatWaktu($permission['start_time']) . ' - ' . formatWaktu($permission['end_time']); ?>
                                    </td>
                                    <td class="px-4 py-3 text-sm text-gray-900 max-w-xs">
                                        <div class="truncate" title="<?php echo escape($permission['purpose']); ?>">
                                            <?php echo escape(substr($permission['purpose'], 0, 40)) . (strlen($permission['purpose']) > 40 ? '...' : ''); ?>
                                        </div>
                                    </td>
                                    <td class="px-4 py-3 whitespace-nowrap">
                                        <?php echo getStatusBadge($permission['status']); ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        <?php else: ?>
                        <div class="p-12 text-center">
                            <i data-feather="inbox" class="w-16 h-16 mx-auto text-gray-300 mb-4"></i>
                            <h3 class="text-lg font-medium text-gray-900 mb-2">Tidak ada data</h3>
                            <p class="text-gray-500">Tidak ada pengajuan izin pada periode yang dipilih.</p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script>
        feather.replace();
        
        // Status Chart
        const statusCtx = document.getElementById('statusChart').getContext('2d');
        new Chart(statusCtx, {
            type: 'doughnut',
            data: {
                labels: ['Disetujui', 'Menunggu', 'Ditolak'],
                datasets: [{
                    data: [<?php echo $stats['approved']; ?>, <?php echo $stats['pending']; ?>, <?php echo $stats['rejected']; ?>],
                    backgroundColor: ['#10b981', '#f59e0b', '#ef4444'],
                    borderWidth: 0
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
        
        // Type Chart
        const typeCtx = document.getElementById('typeChart').getContext('2d');
        new Chart(typeCtx, {
            type: 'bar',
            data: {
                labels: [<?php echo '"' . implode('", "', array_keys($by_type)) . '"'; ?>],
                datasets: [{
                    label: 'Jumlah',
                    data: [<?php echo implode(', ', array_values($by_type)); ?>],
                    backgroundColor: '#3b82f6',
                    borderWidth: 0
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
        
        // Mobile menu toggle
        function toggleSidebar() {
            document.querySelector('.sidebar').classList.toggle('hidden');
        }
    </script>
</body>
</html>