<?php
/**
 * Functions for Permission Management
 * SISKIM - Sistem Izin Keluar Masuk
 */

require_once __DIR__ . '/../config/config.php';

class PermissionManager {
    private $pdo;
    
    public function __construct() {
        global $pdo;
        $this->pdo = $pdo;
    }
    
    /**
     * Get all permission types
     */
    public function getPermissionTypes() {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM permission_types WHERE is_active = 1 ORDER BY name");
            $stmt->execute();
            return $stmt->fetchAll();
        } catch(PDOException $e) {
            return [];
        }
    }
    
    /**
     * Create new permission request
     */
    public function createPermission($data) {
        try {
            // Validasi input
            $required_fields = ['permission_type_id', 'request_date', 'start_time', 'end_time', 'purpose', 'destination'];
            foreach ($required_fields as $field) {
                if (empty($data[$field])) {
                    return [
                        'success' => false,
                        'message' => "Field $field harus diisi"
                    ];
                }
            }
            
            // Validasi tanggal tidak boleh di masa lalu
            if (strtotime($data['request_date']) < strtotime(date('Y-m-d'))) {
                return [
                    'success' => false,
                    'message' => 'Tanggal pengajuan tidak boleh di masa lalu'
                ];
            }
            
            // Validasi waktu
            if (strtotime($data['end_time']) <= strtotime($data['start_time'])) {
                return [
                    'success' => false,
                    'message' => 'Waktu kembali harus setelah waktu keluar'
                ];
            }
            
            // Check apakah sudah ada pengajuan di tanggal dan waktu yang sama
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) FROM permissions 
                WHERE user_id = ? AND request_date = ? AND status IN ('pending', 'approved')
                AND (
                    (start_time <= ? AND end_time >= ?) OR
                    (start_time <= ? AND end_time >= ?) OR
                    (start_time >= ? AND end_time <= ?)
                )
            ");
            $stmt->execute([
                $data['user_id'], 
                $data['request_date'],
                $data['start_time'], $data['start_time'],
                $data['end_time'], $data['end_time'],
                $data['start_time'], $data['end_time']
            ]);
            
            if ($stmt->fetchColumn() > 0) {
                return [
                    'success' => false,
                    'message' => 'Sudah ada pengajuan izin pada waktu tersebut'
                ];
            }
            
            // Insert new permission
            $stmt = $this->pdo->prepare("
                INSERT INTO permissions (
                    user_id, permission_type_id, request_date, start_time, end_time, 
                    purpose, destination, document_path, notes
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $result = $stmt->execute([
                $data['user_id'],
                $data['permission_type_id'],
                $data['request_date'],
                $data['start_time'],
                $data['end_time'],
                $data['purpose'],
                $data['destination'],
                $data['document_path'] ?? null,
                $data['notes'] ?? null
            ]);
            
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Pengajuan izin berhasil dibuat',
                    'permission_id' => $this->pdo->lastInsertId()
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Gagal membuat pengajuan izin'
                ];
            }
            
        } catch(PDOException $e) {
            return [
                'success' => false,
                'message' => 'Terjadi kesalahan sistem: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get permissions by user
     */
    public function getPermissionsByUser($user_id, $page = 1, $limit = 10, $filters = []) {
        try {
            $offset = ($page - 1) * $limit;
            
            // Build where clause
            $where_conditions = ['p.user_id = ?'];
            $params = [$user_id];
            
            if (!empty($filters['status'])) {
                $where_conditions[] = 'p.status = ?';
                $params[] = $filters['status'];
            }
            
            if (!empty($filters['permission_type_id'])) {
                $where_conditions[] = 'p.permission_type_id = ?';
                $params[] = $filters['permission_type_id'];
            }
            
            if (!empty($filters['date_from'])) {
                $where_conditions[] = 'p.request_date >= ?';
                $params[] = $filters['date_from'];
            }
            
            if (!empty($filters['date_to'])) {
                $where_conditions[] = 'p.request_date <= ?';
                $params[] = $filters['date_to'];
            }
            
            $where_clause = implode(' AND ', $where_conditions);
            
            // Get total count
            $count_sql = "
                SELECT COUNT(*) 
                FROM permissions p 
                LEFT JOIN permission_types pt ON p.permission_type_id = pt.id 
                WHERE $where_clause
            ";
            $stmt = $this->pdo->prepare($count_sql);
            $stmt->execute($params);
            $total = $stmt->fetchColumn();
            
            // Get data
            $sql = "
                SELECT p.*, pt.name as permission_type_name, u.name as approved_by_name
                FROM permissions p
                LEFT JOIN permission_types pt ON p.permission_type_id = pt.id
                LEFT JOIN users u ON p.approved_by = u.id
                WHERE $where_clause
                ORDER BY p.created_at DESC
                LIMIT ? OFFSET ?
            ";
            
            $params[] = $limit;
            $params[] = $offset;
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            $permissions = $stmt->fetchAll();
            
            return [
                'success' => true,
                'data' => $permissions,
                'total' => $total,
                'page' => $page,
                'limit' => $limit,
                'total_pages' => ceil($total / $limit)
            ];
            
        } catch(PDOException $e) {
            return [
                'success' => false,
                'message' => 'Terjadi kesalahan sistem: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get all permissions (for admin)
     */
    public function getAllPermissions($page = 1, $limit = 10, $filters = []) {
        try {
            $offset = ($page - 1) * $limit;
            
            // Build where clause
            $where_conditions = ['1=1'];
            $params = [];
            
            if (!empty($filters['status'])) {
                $where_conditions[] = 'p.status = ?';
                $params[] = $filters['status'];
            }
            
            if (!empty($filters['user_id'])) {
                $where_conditions[] = 'p.user_id = ?';
                $params[] = $filters['user_id'];
            }
            
            if (!empty($filters['permission_type_id'])) {
                $where_conditions[] = 'p.permission_type_id = ?';
                $params[] = $filters['permission_type_id'];
            }
            
            if (!empty($filters['date_from'])) {
                $where_conditions[] = 'p.request_date >= ?';
                $params[] = $filters['date_from'];
            }
            
            if (!empty($filters['date_to'])) {
                $where_conditions[] = 'p.request_date <= ?';
                $params[] = $filters['date_to'];
            }
            
            $where_clause = implode(' AND ', $where_conditions);
            
            // Get total count
            $count_sql = "
                SELECT COUNT(*) 
                FROM permissions p 
                WHERE $where_clause
            ";
            $stmt = $this->pdo->prepare($count_sql);
            $stmt->execute($params);
            $total = $stmt->fetchColumn();
            
            // Get data
            $sql = "
                SELECT p.*, pt.name as permission_type_name, 
                       u.name as user_name, u.nip, u.position, u.department,
                       approver.name as approved_by_name
                FROM permissions p
                LEFT JOIN permission_types pt ON p.permission_type_id = pt.id
                LEFT JOIN users u ON p.user_id = u.id
                LEFT JOIN users approver ON p.approved_by = approver.id
                WHERE $where_clause
                ORDER BY p.created_at DESC
                LIMIT ? OFFSET ?
            ";
            
            $params[] = $limit;
            $params[] = $offset;
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            $permissions = $stmt->fetchAll();
            
            return [
                'success' => true,
                'data' => $permissions,
                'total' => $total,
                'page' => $page,
                'limit' => $limit,
                'total_pages' => ceil($total / $limit)
            ];
            
        } catch(PDOException $e) {
            return [
                'success' => false,
                'message' => 'Terjadi kesalahan sistem: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get permission by ID
     */
    public function getPermissionById($id, $user_id = null) {
        try {
            $sql = "
                SELECT p.*, pt.name as permission_type_name, 
                       u.name as user_name, u.nip, u.position, u.department, u.email, u.phone,
                       approver.name as approved_by_name
                FROM permissions p
                LEFT JOIN permission_types pt ON p.permission_type_id = pt.id
                LEFT JOIN users u ON p.user_id = u.id
                LEFT JOIN users approver ON p.approved_by = approver.id
                WHERE p.id = ?
            ";
            
            $params = [$id];
            
            // Jika bukan admin, hanya bisa lihat izin sendiri
            if ($user_id) {
                $sql .= " AND p.user_id = ?";
                $params[] = $user_id;
            }
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            $permission = $stmt->fetch();
            
            if ($permission) {
                return [
                    'success' => true,
                    'data' => $permission
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Data pengajuan tidak ditemukan'
                ];
            }
            
        } catch(PDOException $e) {
            return [
                'success' => false,
                'message' => 'Terjadi kesalahan sistem: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Update permission status (approve/reject)
     */
    public function updatePermissionStatus($id, $status, $approved_by, $notes = '') {
        try {
            if (!in_array($status, ['approved', 'rejected'])) {
                return [
                    'success' => false,
                    'message' => 'Status tidak valid'
                ];
            }
            
            // Check if permission exists and is pending
            $stmt = $this->pdo->prepare("SELECT status FROM permissions WHERE id = ?");
            $stmt->execute([$id]);
            $current_status = $stmt->fetchColumn();
            
            if (!$current_status) {
                return [
                    'success' => false,
                    'message' => 'Pengajuan tidak ditemukan'
                ];
            }
            
            if ($current_status !== 'pending') {
                return [
                    'success' => false,
                    'message' => 'Pengajuan sudah diproses sebelumnya'
                ];
            }
            
            // Update status
            $stmt = $this->pdo->prepare("
                UPDATE permissions 
                SET status = ?, approved_by = ?, approved_at = CURRENT_TIMESTAMP, 
                    rejection_reason = ?, updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            
            $rejection_reason = ($status === 'rejected') ? $notes : null;
            
            $result = $stmt->execute([$status, $approved_by, $rejection_reason, $id]);
            
            if ($result) {
                $status_text = ($status === 'approved') ? 'disetujui' : 'ditolak';
                return [
                    'success' => true,
                    'message' => "Pengajuan berhasil $status_text"
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Gagal memperbarui status pengajuan'
                ];
            }
            
        } catch(PDOException $e) {
            return [
                'success' => false,
                'message' => 'Terjadi kesalahan sistem: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Delete permission (only for pending status and own permission)
     */
    public function deletePermission($id, $user_id) {
        try {
            // Check if permission exists, is pending, and belongs to user
            $stmt = $this->pdo->prepare("
                SELECT status, document_path 
                FROM permissions 
                WHERE id = ? AND user_id = ?
            ");
            $stmt->execute([$id, $user_id]);
            $permission = $stmt->fetch();
            
            if (!$permission) {
                return [
                    'success' => false,
                    'message' => 'Pengajuan tidak ditemukan'
                ];
            }
            
            if ($permission['status'] !== 'pending') {
                return [
                    'success' => false,
                    'message' => 'Hanya pengajuan yang masih menunggu yang bisa dihapus'
                ];
            }
            
            // Delete file if exists
            if ($permission['document_path']) {
                deleteFile($permission['document_path']);
            }
            
            // Delete permission
            $stmt = $this->pdo->prepare("DELETE FROM permissions WHERE id = ? AND user_id = ?");
            $result = $stmt->execute([$id, $user_id]);
            
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Pengajuan berhasil dihapus'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Gagal menghapus pengajuan'
                ];
            }
            
        } catch(PDOException $e) {
            return [
                'success' => false,
                'message' => 'Terjadi kesalahan sistem: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get permissions for calendar
     */
    public function getPermissionsForCalendar($month, $year, $user_id = null) {
        try {
            $sql = "
                SELECT p.*, pt.name as permission_type_name, u.name as user_name
                FROM permissions p
                LEFT JOIN permission_types pt ON p.permission_type_id = pt.id
                LEFT JOIN users u ON p.user_id = u.id
                WHERE MONTH(p.request_date) = ? AND YEAR(p.request_date) = ?
                AND p.status = 'approved'
            ";
            
            $params = [$month, $year];
            
            if ($user_id) {
                $sql .= " AND p.user_id = ?";
                $params[] = $user_id;
            }
            
            $sql .= " ORDER BY p.request_date, p.start_time";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            
            return $stmt->fetchAll();
            
        } catch(PDOException $e) {
            return [];
        }
    }
    
    /**
     * Get statistics
     */
    public function getStatistics($user_id = null, $date_range = 'month') {
        try {
            $date_condition = '';
            switch ($date_range) {
                case 'today':
                    $date_condition = "AND p.request_date = CURRENT_DATE";
                    break;
                case 'week':
                    $date_condition = "AND p.request_date >= date('now', 'weekday 0', '-6 days') AND p.request_date <= date('now', 'weekday 0')";
                    break;
                case 'month':
                    $date_condition = "AND strftime('%Y-%m', p.request_date) = strftime('%Y-%m', 'now')";
                    break;
                case 'year':
                    $date_condition = "AND strftime('%Y', p.request_date) = strftime('%Y', 'now')";
                    break;
            }
            
            $user_condition = $user_id ? "AND p.user_id = $user_id" : "";
            
            $stats = [];
            
            // Total requests
            $stmt = $this->pdo->query("
                SELECT COUNT(*) FROM permissions p 
                WHERE 1=1 $date_condition $user_condition
            ");
            $stats['total'] = $stmt->fetchColumn();
            
            // By status
            foreach (['pending', 'approved', 'rejected'] as $status) {
                $stmt = $this->pdo->query("
                    SELECT COUNT(*) FROM permissions p 
                    WHERE p.status = '$status' $date_condition $user_condition
                ");
                $stats[$status] = $stmt->fetchColumn();
            }
            
            return $stats;
            
        } catch(PDOException $e) {
            return [
                'total' => 0,
                'pending' => 0,
                'approved' => 0,
                'rejected' => 0
            ];
        }
    }
}

// Instance global
$permissionManager = new PermissionManager();
?>