<?php
require_once 'config/config.php';

// Require admin access untuk upload
if (isset($_POST['upload']) && !$auth->isAdmin()) {
    setFlashMessage('error', 'Akses ditolak. Hanya admin yang dapat mengupload gambar.');
    header('Location: gallery.php');
    exit();
}

$page_title = 'Galeri Gambar';
$current_user = $auth->getCurrentUser();

// Handle file upload
if (isset($_POST['upload']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
    $upload_dir = __DIR__ . '/gambar/';
    
    // Buat folder jika belum ada
    if (!file_exists($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    $file = $_FILES['image'];
    $filename = $file['name'];
    $temp_name = $file['tmp_name'];
    $file_size = $file['size'];
    
    // Validasi ekstensi file
    $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif'];
    $file_extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    
    if (!in_array($file_extension, $allowed_extensions)) {
        setFlashMessage('error', 'Format file tidak didukung. Gunakan JPG, PNG, atau GIF.');
    } elseif ($file_size > MAX_FILE_SIZE) {
        setFlashMessage('error', 'Ukuran file terlalu besar. Maksimal ' . formatFileSize(MAX_FILE_SIZE));
    } else {
        // Generate nama file unik
        $new_filename = time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '', $filename);
        $target_path = $upload_dir . $new_filename;
        
        if (move_uploaded_file($temp_name, $target_path)) {
            setFlashMessage('success', 'Gambar berhasil diupload: ' . $new_filename);
        } else {
            setFlashMessage('error', 'Gagal mengupload gambar.');
        }
    }
}

// Handle delete
if (isset($_POST['delete']) && $auth->isAdmin()) {
    $filename = $_POST['filename'];
    $filepath = __DIR__ . '/gambar/' . $filename;
    
    // Jangan hapus gambar sistem
    $system_images = ['Logo_pnwtp.png', 'kop_surat.png', 'gedung-pn-wtp.png'];
    
    if (in_array($filename, $system_images)) {
        setFlashMessage('error', 'Tidak dapat menghapus gambar sistem.');
    } elseif (file_exists($filepath)) {
        if (unlink($filepath)) {
            setFlashMessage('success', 'Gambar berhasil dihapus.');
        } else {
            setFlashMessage('error', 'Gagal menghapus gambar.');
        }
    }
}

// Get all images from gambar folder
$image_dir = __DIR__ . '/gambar/';
$images = [];

if (is_dir($image_dir)) {
    $files = scandir($image_dir);
    foreach ($files as $file) {
        if ($file !== '.' && $file !== '..') {
            $filepath = $image_dir . $file;
            if (is_file($filepath)) {
                $info = pathinfo($filepath);
                if (in_array(strtolower($info['extension']), ['jpg', 'jpeg', 'png', 'gif'])) {
                    $images[] = [
                        'filename' => $file,
                        'size' => filesize($filepath),
                        'modified' => filemtime($filepath),
                        'is_system' => in_array($file, ['Logo_pnwtp.png', 'kop_surat.png', 'gedung-pn-wtp.png'])
                    ];
                }
            }
        }
    }
}

// Sort by modification time (newest first)
usort($images, function($a, $b) {
    return $b['modified'] - $a['modified'];
});
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - SISKIM PN Watampone</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/feather-icons/dist/feather.min.js"></script>
    <script src="https://unpkg.com/feather-icons"></script>
    <style>
        .sidebar { transition: all 0.3s; }
        .nav-item.active { background-color: #1e3a8a; color: white; }
        .nav-item.active svg { color: white; }
    </style>
</head>
<body class="bg-gray-50">
    <div class="flex h-screen overflow-hidden">
        <!-- Sidebar -->
        <?php include 'includes/sidebar.php'; ?>

        <!-- Main Content -->
        <div class="flex-1 overflow-auto">
            <!-- Header -->
            <?php include 'includes/header.php'; ?>

            <!-- Content -->
            <main class="p-6">
                <!-- Flash Messages -->
                <?php showFlashMessage(); ?>
                
                <div class="mb-6">
                    <h1 class="text-2xl font-bold text-gray-800">Galeri Gambar</h1>
                    <p class="text-gray-600 mt-1">Kelola gambar untuk sistem SISKIM</p>
                </div>

                <!-- Upload Section (Admin Only) -->
                <?php if ($auth->isAdmin()): ?>
                <div class="bg-white rounded-lg shadow p-6 mb-6">
                    <h2 class="text-lg font-semibold mb-4">Upload Gambar Baru</h2>
                    <form method="POST" enctype="multipart/form-data" class="flex items-end space-x-4">
                        <div class="flex-1">
                            <label class="block text-sm font-medium text-gray-700 mb-2">Pilih File Gambar</label>
                            <input type="file" name="image" accept="image/*" required
                                   class="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100">
                        </div>
                        <div>
                            <button type="submit" name="upload" class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 flex items-center">
                                <i data-feather="upload" class="w-4 h-4 mr-2"></i>
                                Upload
                            </button>
                        </div>
                    </form>
                    <p class="text-xs text-gray-500 mt-2">
                        Format yang didukung: JPG, PNG, GIF. Maksimal <?php echo formatFileSize(MAX_FILE_SIZE); ?>
                    </p>
                </div>
                <?php endif; ?>

                <!-- Images Grid -->
                <div class="bg-white rounded-lg shadow overflow-hidden">
                    <div class="px-6 py-4 border-b border-gray-200">
                        <h2 class="text-lg font-semibold">Daftar Gambar (<?php echo count($images); ?>)</h2>
                    </div>
                    
                    <?php if (!empty($images)): ?>
                    <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 xl:grid-cols-6 gap-4 p-6">
                        <?php foreach ($images as $image): ?>
                        <div class="relative group border border-gray-200 rounded-lg overflow-hidden bg-white shadow-sm hover:shadow-md transition-shadow">
                            <div class="aspect-square bg-gray-100 flex items-center justify-center overflow-hidden">
                                <img src="<?php echo getImagePath($image['filename']); ?>" 
                                     alt="<?php echo escape($image['filename']); ?>"
                                     class="w-full h-full object-cover"
                                     loading="lazy">
                            </div>
                            <div class="p-3">
                                <h3 class="text-sm font-medium text-gray-900 truncate" title="<?php echo escape($image['filename']); ?>">
                                    <?php echo escape($image['filename']); ?>
                                </h3>
                                <div class="text-xs text-gray-500 mt-1">
                                    <div><?php echo formatFileSize($image['size']); ?></div>
                                    <div><?php echo date('d/m/Y H:i', $image['modified']); ?></div>
                                    <?php if ($image['is_system']): ?>
                                    <span class="inline-block bg-blue-100 text-blue-800 text-xs px-2 py-1 rounded-full mt-1">Sistem</span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <!-- Action buttons -->
                            <div class="absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity space-y-1">
                                <button onclick="viewImage('<?php echo getImagePath($image['filename']); ?>', '<?php echo escape($image['filename']); ?>')"
                                        class="block bg-white bg-opacity-90 hover:bg-opacity-100 p-1.5 rounded text-blue-600">
                                    <i data-feather="eye" class="w-4 h-4"></i>
                                </button>
                                <button onclick="copyImageUrl('<?php echo getImageUrl($image['filename']); ?>')"
                                        class="block bg-white bg-opacity-90 hover:bg-opacity-100 p-1.5 rounded text-green-600">
                                    <i data-feather="copy" class="w-4 h-4"></i>
                                </button>
                                <?php if ($auth->isAdmin() && !$image['is_system']): ?>
                                <button onclick="deleteImage('<?php echo escape($image['filename']); ?>')"
                                        class="block bg-white bg-opacity-90 hover:bg-opacity-100 p-1.5 rounded text-red-600">
                                    <i data-feather="trash-2" class="w-4 h-4"></i>
                                </button>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <?php else: ?>
                    <div class="p-12 text-center">
                        <i data-feather="image" class="w-16 h-16 mx-auto text-gray-300 mb-4"></i>
                        <h3 class="text-lg font-medium text-gray-900 mb-2">Belum ada gambar</h3>
                        <p class="text-gray-500">Upload gambar untuk memulai.</p>
                    </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>

    <!-- Image Viewer Modal -->
    <div id="imageModal" class="hidden fixed inset-0 bg-black bg-opacity-75 z-50 flex items-center justify-center">
        <div class="relative max-w-4xl max-h-full p-4">
            <button onclick="closeImageModal()" class="absolute top-4 right-4 text-white hover:text-gray-300 z-10">
                <i data-feather="x" class="w-8 h-8"></i>
            </button>
            <img id="modalImage" src="" alt="" class="max-w-full max-h-full object-contain">
            <div class="absolute bottom-4 left-4 bg-black bg-opacity-50 text-white px-3 py-1 rounded">
                <span id="modalImageName"></span>
            </div>
        </div>
    </div>

    <script>
        feather.replace();
        
        function viewImage(imageSrc, imageName) {
            document.getElementById('modalImage').src = imageSrc;
            document.getElementById('modalImageName').textContent = imageName;
            document.getElementById('imageModal').classList.remove('hidden');
        }
        
        function closeImageModal() {
            document.getElementById('imageModal').classList.add('hidden');
        }
        
        function copyImageUrl(url) {
            navigator.clipboard.writeText(url).then(function() {
                alert('URL gambar berhasil disalin ke clipboard!');
            });
        }
        
        function deleteImage(filename) {
            if (confirm('Yakin ingin menghapus gambar "' + filename + '"?')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = `
                    <input type="hidden" name="delete" value="1">
                    <input type="hidden" name="filename" value="${filename}">
                `;
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        // Close modal on escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeImageModal();
            }
        });
        
        // Close modal on background click
        document.getElementById('imageModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeImageModal();
            }
        });
    </script>
</body>
</html>