<?php
/**
 * Sistem Autentikasi
 * SISKIM - Sistem Izin Keluar Masuk
 * Pengadilan Negeri Watampone
 */

// Start session jika belum dimulai
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . '/database.php';

class Auth {
    private $db;
    private $pdo;
    
    public function __construct() {
        global $database, $pdo;
        $this->db = $database;
        $this->pdo = $pdo;
    }
    
    /**
     * Login user
     */
    public function login($nip, $password) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT id, nip, name, email, password, role, position, department, phone, is_active 
                FROM users 
                WHERE nip = ? AND is_active = 1
            ");
            $stmt->execute([$nip]);
            $user = $stmt->fetch();
            
            if ($user && password_verify($password, $user['password'])) {
                // Set session
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['user_nip'] = $user['nip'];
                $_SESSION['user_name'] = $user['name'];
                $_SESSION['user_email'] = $user['email'];
                $_SESSION['user_role'] = $user['role'];
                $_SESSION['user_position'] = $user['position'];
                $_SESSION['user_department'] = $user['department'];
                $_SESSION['user_phone'] = $user['phone'];
                $_SESSION['login_time'] = time();
                
                // Update last login
                $this->updateLastLogin($user['id']);
                
                return [
                    'success' => true,
                    'message' => 'Login berhasil',
                    'user' => $user
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'NIP atau password salah'
                ];
            }
        } catch(PDOException $e) {
            return [
                'success' => false,
                'message' => 'Terjadi kesalahan sistem: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Logout user
     */
    public function logout() {
        // Hapus semua session data
        session_unset();
        session_destroy();
        
        // Redirect ke login
        header('Location: login.php');
        exit();
    }
    
    /**
     * Cek apakah user sudah login
     */
    public function isLoggedIn() {
        return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
    }
    
    /**
     * Cek apakah user adalah admin
     */
    public function isAdmin() {
        return $this->isLoggedIn() && $_SESSION['user_role'] === 'admin';
    }
    
    /**
     * Get current user data
     */
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        return [
            'id' => $_SESSION['user_id'],
            'nip' => $_SESSION['user_nip'],
            'name' => $_SESSION['user_name'],
            'email' => $_SESSION['user_email'],
            'role' => $_SESSION['user_role'],
            'position' => $_SESSION['user_position'],
            'department' => $_SESSION['user_department'],
            'phone' => $_SESSION['user_phone']
        ];
    }
    
    /**
     * Require login - redirect jika belum login
     */
    public function requireLogin() {
        if (!$this->isLoggedIn()) {
            header('Location: login.php');
            exit();
        }
    }
    
    /**
     * Require admin - redirect jika bukan admin
     */
    public function requireAdmin() {
        $this->requireLogin();
        if (!$this->isAdmin()) {
            header('Location: dashboard.php?error=access_denied');
            exit();
        }
    }
    
    /**
     * Update last login time
     */
    private function updateLastLogin($user_id) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE users 
                SET updated_at = CURRENT_TIMESTAMP 
                WHERE id = ?
            ");
            $stmt->execute([$user_id]);
        } catch(PDOException $e) {
            // Log error tapi jangan stop proses login
            error_log("Update last login failed: " . $e->getMessage());
        }
    }
    
    /**
     * Register user baru (untuk admin)
     */
    public function register($data) {
        try {
            // Validasi input
            $required_fields = ['nip', 'name', 'password'];
            foreach ($required_fields as $field) {
                if (empty($data[$field])) {
                    return [
                        'success' => false,
                        'message' => "Field $field harus diisi"
                    ];
                }
            }
            
            // Cek apakah NIP sudah ada
            $stmt = $this->pdo->prepare("SELECT COUNT(*) FROM users WHERE nip = ?");
            $stmt->execute([$data['nip']]);
            if ($stmt->fetchColumn() > 0) {
                return [
                    'success' => false,
                    'message' => 'NIP sudah terdaftar'
                ];
            }
            
            // Insert user baru
            $stmt = $this->pdo->prepare("
                INSERT INTO users (nip, name, email, password, role, position, department, phone) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $result = $stmt->execute([
                $data['nip'],
                $data['name'],
                $data['email'] ?? null,
                password_hash($data['password'], PASSWORD_DEFAULT),
                $data['role'] ?? 'user',
                $data['position'] ?? null,
                $data['department'] ?? null,
                $data['phone'] ?? null
            ]);
            
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'User berhasil didaftarkan',
                    'user_id' => $this->pdo->lastInsertId()
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Gagal mendaftarkan user'
                ];
            }
            
        } catch(PDOException $e) {
            return [
                'success' => false,
                'message' => 'Terjadi kesalahan sistem: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Change password
     */
    public function changePassword($user_id, $old_password, $new_password) {
        try {
            // Ambil password lama
            $stmt = $this->pdo->prepare("SELECT password FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $current_password = $stmt->fetchColumn();
            
            if (!password_verify($old_password, $current_password)) {
                return [
                    'success' => false,
                    'message' => 'Password lama tidak sesuai'
                ];
            }
            
            // Update password baru
            $stmt = $this->pdo->prepare("
                UPDATE users 
                SET password = ?, updated_at = CURRENT_TIMESTAMP 
                WHERE id = ?
            ");
            
            $result = $stmt->execute([
                password_hash($new_password, PASSWORD_DEFAULT),
                $user_id
            ]);
            
            if ($result) {
                return [
                    'success' => true,
                    'message' => 'Password berhasil diubah'
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Gagal mengubah password'
                ];
            }
            
        } catch(PDOException $e) {
            return [
                'success' => false,
                'message' => 'Terjadi kesalahan sistem: ' . $e->getMessage()
            ];
        }
    }
}

// Instance global auth
$auth = new Auth();

/**
 * Helper functions
 */

// Fungsi untuk generate CSRF token
function generateCSRFToken() {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Fungsi untuk validasi CSRF token
function validateCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Fungsi untuk sanitize input
function sanitize($input) {
    return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
}

// Fungsi untuk escape output
function escape($output) {
    return htmlspecialchars($output, ENT_QUOTES, 'UTF-8');
}
?>